import {PureComponent, ReactNode} from 'react';
import {withRouter, RouteComponentProps} from 'react-router-dom';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';
import {IHotelPartnerOffer} from 'types/hotels/offer/IHotelOffer';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {reachGoal} from 'utilities/metrika';
import {getViewModePageParamsByQuery} from 'projects/hotels/utilities/getHotelPageParams/getHotelPageParams';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */
import PartnerOffer from './components/PartnerOffer/PartnerOffer';

/* Component Types */
interface IHotelPageOffersProps
    extends IWithClassName,
        IWithQaAttributes,
        RouteComponentProps {
    hasMainOffers: boolean;
    offers: IHotelPartnerOffer[];
    nightsCount?: number;
    operatorById: OperatorByIdType;
    isRoomGroup?: boolean;
    isShortOperator?: boolean;
    onOfferSelect?: () => void;
}

class PartnerOffers extends PureComponent<IHotelPageOffersProps> {
    private handleOfferSelect = (): void => {
        const {onOfferSelect} = this.props;

        reachGoal(EHotelsGoal.HOTEL_PAGE_PARTNER_REDIRECT);

        if (onOfferSelect) {
            onOfferSelect();
        }
    };

    /* Render */

    private renderOffer = (
        offer: IHotelPartnerOffer,
        index: number,
    ): ReactNode => {
        const {location, operatorById, hasMainOffers, nightsCount} = this.props;
        const queryByLocation = getQueryByLocation(location);
        const {isHeadless} = getViewModePageParamsByQuery(queryByLocation);

        return (
            <PartnerOffer
                isHeadless={isHeadless}
                key={offer.operatorId}
                operatorById={operatorById}
                onOfferSelect={this.handleOfferSelect}
                hasMainOffers={hasMainOffers}
                offer={offer}
                nightsCount={nightsCount}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'offer',
                    key: index,
                })}
            />
        );
    };

    render(): ReactNode {
        const {className, offers} = this.props;

        return (
            <div className={className} {...prepareQaAttributes(this.props)}>
                {offers.map(this.renderOffer)}
            </div>
        );
    }
}

export default withRouter(PartnerOffers);
