import React, {useCallback} from 'react';

import {EAppActions} from 'constants/platforms/TPlatforms';

import {IHotelPartnerOffer} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-HotelPagePartnerOffersInfo';

import HotelOperator from 'projects/hotels/components/HotelOperator/HotelOperator';
import Offer from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/Offer/Offer';
import OfferMealInfo from 'projects/hotels/components/OfferMealInfo/OfferMealInfo';
import HotelsCancellationInfo from 'projects/hotels/components/HotelsCancellationInfo/HotelsCancellationInfo';

import {useCoordinator} from 'contexts/PlatformContext';

import cx from './PartnerOffer.scss';

export interface IHotelPageOfferMobileProps
    extends IWithClassName,
        IWithQaAttributes {
    hasMainOffers: boolean;
    nightsCount?: number;
    offer: IHotelPartnerOffer;
    operatorById: OperatorByIdType;
    onOfferSelect: () => void;
    isHeadless?: boolean;
}

const PartnerOffer: React.FC<IHotelPageOfferMobileProps> = ({
    className,
    offer,
    operatorById,
    isHeadless,
    onOfferSelect,
    hasMainOffers,
    nightsCount,
    ...rest
}) => {
    const {
        operatorId,
        defaultOffer,
        defaultOfferCancellationInfo,
        defaultOfferPansion,
    } = offer;

    const coordinator = useCoordinator();
    const deviceType = useDeviceType();

    const handleOfferSelect: React.MouseEventHandler = useCallback(
        e => {
            const processed = coordinator.doAction(
                EAppActions.NAVIGATE_TO_EXTERNAL_URL,
                defaultOffer.landingUrl,
            );

            if (processed) {
                e.preventDefault();
            }

            onOfferSelect?.();
        },
        [coordinator, defaultOffer.landingUrl, onOfferSelect],
    );

    const hotelOperator = (
        <HotelOperator
            iconClassName={cx('icon')}
            operator={operatorById[operatorId]}
            partnerUrl={defaultOffer.landingUrl}
            theme="black"
            size="l"
            onClick={handleOfferSelect}
            isHeadless={isHeadless}
            {...prepareQaAttributes({
                parent: rest,
                current: 'hotelOperator',
            })}
        />
    );

    return (
        <Offer
            className={cx(className, 'offer', deviceMods('offer', deviceType))}
            isHeadless={isHeadless}
            title={hotelOperator}
            labels={
                <>
                    {defaultOfferPansion && (
                        <OfferMealInfo
                            size="m"
                            isIconHidden
                            mealInfo={defaultOfferPansion}
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'offerLabels-mealInfo',
                            })}
                        />
                    )}
                    {defaultOfferCancellationInfo && (
                        <HotelsCancellationInfo
                            cancellationInfo={defaultOfferCancellationInfo}
                            {...prepareQaAttributes({
                                parent: rest,
                                current: 'offerLabels-cancellationInfo',
                            })}
                        />
                    )}
                </>
            }
            nightsCount={nightsCount}
            onOfferSelect={handleOfferSelect}
            price={defaultOffer.price}
            discountInfo={defaultOffer.discountInfo}
            plusInfo={defaultOffer?.offerYandexPlusInfo}
            buttonTheme={hasMainOffers ? 'secondary' : 'primary'}
            buttonText={i18nBlock.selectAction()}
            buttonUrl={defaultOffer.landingUrl}
            type="horizontal"
            isPartner
            showNightsCount={Boolean(nightsCount)}
            linkTarget="_blank"
            {...prepareQaAttributes(rest)}
        />
    );
};

export default PartnerOffer;
