import {FunctionComponent, ReactNode, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {
    IHotelBoyMetaRoomWithOffers,
    IHotelOffersInfo,
} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';

import hotelRoomsBoyMetaSelector from 'selectors/hotels/hotel/common/hotelRoomsBoyMetaSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import Room from './components/Room/Room';
import RoomsWithoutOffers from './components/RoomsWithoutOffers/RoomsWithoutOffers';

import cx from './Rooms.scss';

interface IHotelPageRoomsProps extends IWithClassName, IWithQaAttributes {
    offersInfo: IHotelOffersInfo;
    nightsCount?: number;
    canShowImages?: boolean;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onCheckPricesClick?: () => void;
    chosenRoomId?: string;
    searchParams?: IRequiredOfferParams;
}

const Rooms: FunctionComponent<IHotelPageRoomsProps> = ({
    onOfferSelect,
    canShowImages = true,
    nightsCount,
    className,
    onOfferWatchButtonClick,
    onCheckPricesClick,
    chosenRoomId,
    searchParams,
    offersInfo,
    ...rest
}) => {
    const rooms = useSelector(hotelRoomsBoyMetaSelector);
    const deviceType = useDeviceType();
    const {partnerOffers, operatorById} = offersInfo;

    const {roomsWithOffers, roomsWithoutOffers} = useMemo(
        () =>
            rooms.reduce<{
                roomsWithOffers: IHotelBoyMetaRoomWithOffers[];
                roomsWithoutOffers: IHotelBoyMetaRoomWithOffers[];
            }>(
                (result, room) => {
                    if (room.offers.length) {
                        result.roomsWithOffers.push(room);
                    } else {
                        result.roomsWithoutOffers.push(room);
                    }

                    return result;
                },
                {roomsWithOffers: [], roomsWithoutOffers: []},
            ),
        [rooms],
    );

    const renderRoom = (room: IHotelBoyMetaRoomWithOffers): ReactNode => {
        return (
            <Room
                key={room.id}
                room={room}
                partnerOffers={partnerOffers}
                operatorById={operatorById}
                onOfferSelect={onOfferSelect}
                onOfferWatchButtonClick={onOfferWatchButtonClick}
                onCheckPricesClick={onCheckPricesClick}
                canShowImages={canShowImages}
                nightsCount={nightsCount}
                chosenRoomId={chosenRoomId}
                searchParams={searchParams}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'room',
                    key: room.id,
                })}
            />
        );
    };

    const rootClassName = cx(className, 'root', deviceMods('root', deviceType));

    if (!searchParams) {
        return (
            <div className={rootClassName}>
                {roomsWithoutOffers.map(renderRoom)}
            </div>
        );
    }

    return (
        <>
            <div className={rootClassName}>
                {roomsWithOffers.map(renderRoom)}
            </div>

            {Boolean(roomsWithoutOffers.length) && (
                <RoomsWithoutOffers
                    className={rootClassName}
                    rooms={roomsWithoutOffers}
                    renderRoom={renderRoom}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'roomsWithoutOffers',
                    })}
                />
            )}
        </>
    );
};

export default Rooms;
