import {useRef} from 'react';

import {IRoomAmenity} from 'types/hotels/offer/IHotelRoom';
import {IWithClassName} from 'types/withClassName';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {useBoolean} from 'utilities/hooks/useBoolean';
import getRoomAmenityComponentIconById from 'projects/hotels/utilities/amenities/getRoomAmenityComponentIconById';

import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';

import cx from './Amenity.scss';

export interface IHotelPageRoomAmenityProps extends IWithClassName {
    amenity: IRoomAmenity;
    amenityTextSize: 's' | 'm';
    amenityFontWeight: 'normal' | 'medium' | 'bold';
    hideFeatures?: boolean;
    hasPopup?: boolean;
}

const Amenity: React.FC<IHotelPageRoomAmenityProps> = ({
    className,
    amenity,
    amenityTextSize,
    amenityFontWeight,
    hideFeatures,
    hasPopup,
}) => {
    const {icon, name} = amenity;
    const anchorRef = useRef<HTMLDivElement | null>(null);
    const {
        value: isVisible,
        setTrue: showPopup,
        setFalse: hidePopup,
    } = useBoolean(false);

    if (hideFeatures) {
        const Icon = getRoomAmenityComponentIconById(icon);

        return (
            <>
                <Flex
                    alignItems="center"
                    justifyContent="center"
                    onMouseOver={hasPopup ? showPopup : undefined}
                    onMouseOut={hasPopup ? hidePopup : undefined}
                    className={cx(
                        'amenityBlock',
                        !hasPopup && 'amenityBlock_noPopup',
                    )}
                    innerRef={anchorRef}
                >
                    <Icon />
                </Flex>
                {hasPopup && (
                    <MessageBoxPopup
                        className={cx('popupContent')}
                        isVisible={isVisible}
                        anchorRef={anchorRef}
                        direction={EPopupDirection.BOTTOM}
                        onClose={hidePopup}
                        hasClose={false}
                    >
                        <Text color="inverse">{name}</Text>
                    </MessageBoxPopup>
                )}
            </>
        );
    }

    return (
        <TextWithIcon
            size={amenityTextSize}
            text={name}
            className={cx('amenity', className, {
                [`amenity_fontWeight_${amenityFontWeight}`]:
                    Boolean(amenityFontWeight),
            })}
            iconLeftClassName={cx('amenityIcon')}
            iconLeft={getRoomAmenityComponentIconById(icon)}
        />
    );
};

export default Amenity;
