import {
    ReactNode,
    FunctionComponent,
    useCallback,
    useState,
    useEffect,
    useRef,
} from 'react';

import {HOTEL_ROOM_ANCHOR} from 'constants/urls/anchors';

import {IHotelBoyMetaRoomWithOffers} from 'reducers/hotels/hotelPage/hotelInfo/types';
import {
    IHotelPartnerOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import scrollToNode from 'utilities/dom/scrollToNode';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import getGuestsCount from 'projects/hotels/pages/HotelPage/utilities/getGuestsCount';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nBlock from 'i18n/hotels-HotelPageRooms';
import * as i18nOffersBlock from 'i18n/hotels-HotelPageOffersInfo';

import Card from 'components/Card/Card';
import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Anchor from 'components/Anchor/Anchor';
import Heading from 'components/Heading/Heading';
import Separator from 'components/Separator/Separator';
import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import HotelImageStub from 'projects/hotels/components/HotelImageStub/HotelImageStub';
import HotelOfferBadges from 'projects/hotels/components/HotelOfferBadges/HotelOfferBadges';
import RoomBedGroupsAndSize from 'projects/hotels/components/RoomBedGroupsAndSize/RoomBedGroupsAndSize';
import MainOffers from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/MainOffers/MainOffers';
import HotelPageImagesViewer from 'projects/hotels/pages/HotelPage/components/HotelPageImagesViewer/HotelPageImagesViewer';
import Amenities from 'projects/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/components/Amenities/Amenities';
import HotelPageRoomGallery from 'projects/hotels/pages/HotelPage/components/HotelPageRoomGallery/HotelPageRoomGallery';
import HotelPageRoomDetailedInfoButton from 'projects/hotels/pages/HotelPage/components/HotelPageAllAmenities/blocks/HotelPageRoomDetailedInfo/HotelPageRoomDetailedInfoButton';

import cx from './Room.scss';

const SHORT_AMENITY_LIMIT = 3;

export interface IRoomProps extends IWithQaAttributes {
    room: IHotelBoyMetaRoomWithOffers;
    partnerOffers?: IHotelPartnerOffer[] | null;
    operatorById?: OperatorByIdType;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onCheckPricesClick?: () => void;
    canShowImages?: boolean;
    nightsCount?: number;
    chosenRoomId?: string;
    searchParams?: IRequiredOfferParams;
}

const Room: FunctionComponent<IRoomProps> = ({
    room,
    partnerOffers,
    operatorById,
    canShowImages,
    onOfferSelect,
    onOfferWatchButtonClick,
    onCheckPricesClick,
    nightsCount,
    chosenRoomId,
    searchParams,
    ...rest
}) => {
    const {id, offers, name, images, mainAmenities, area, bedGroups, badges} =
        room;
    const {hotelsHideFeatures} = useExperiments();
    const deviceType = useDeviceType();
    const {isDesktop, isMobile} = deviceType;
    const {
        value: isCheckPricesHovered,
        setTrue: setCheckPricesHovered,
        setFalse: unsetCheckPricesHovered,
    } = useBoolean(false);
    const roomRef = useRef<HTMLElement>(null);
    const checkPricesRef = useRef<HTMLElement>(null);

    const handleWatchButtonClick = useCallback(() => {
        onOfferWatchButtonClick?.(id);
        reachGoal(EHotelsGoal.HOTEL_PAGE_ROOM_DETAILED_FEATURES_SHOW);
    }, [onOfferWatchButtonClick, id]);

    const [modalImagesInitialIndex, setModalImagesInitialIndex] = useState<
        number | undefined
    >(undefined);

    useEffect(() => {
        if (chosenRoomId === id) {
            if (roomRef.current) {
                scrollToNode(roomRef.current, {behavior: 'smooth'});
            }
        }
    }, [chosenRoomId, id, roomRef]);

    const handleCloseModalImages = useCallback(() => {
        setModalImagesInitialIndex(undefined);
    }, [setModalImagesInitialIndex]);

    const handleImageClick = useCallback(
        (imageIndex: number) => {
            setModalImagesInitialIndex(imageIndex);

            reachGoal(EHotelsGoal.HOTEL_PAGE_OFFERS_TAB_IMAGE_CLICK);
        },
        [setModalImagesInitialIndex],
    );

    const renderModalImagesViewer = (): ReactNode => {
        return (
            images && (
                <HotelPageImagesViewer
                    isVisible={typeof modalImagesInitialIndex === 'number'}
                    totalImageCount={images.length}
                    initialIndex={modalImagesInitialIndex}
                    name={name}
                    images={images}
                    onCloseClick={handleCloseModalImages}
                />
            )
        );
    };

    function renderOffers(): ReactNode {
        const hasOffers = offers.length > 0;

        if (!searchParams) {
            return (
                <Flex justifyContent="flex-end">
                    {isMobile && (
                        <Separator className={cx('checkPricesSeparator')} />
                    )}
                    <Button
                        theme="primary"
                        onClick={onCheckPricesClick}
                        onMouseEnter={setCheckPricesHovered}
                        onMouseLeave={unsetCheckPricesHovered}
                        innerRef={checkPricesRef}
                        className={cx('checkPricesButton')}
                        size={isMobile ? 'l' : 'm'}
                    >
                        {i18nOffersBlock.checkPrices()}
                    </Button>
                    {isDesktop && (
                        <MessageBoxPopup
                            isVisible={isCheckPricesHovered}
                            anchorRef={checkPricesRef}
                        >
                            <div className={cx('tooltip')}>
                                {i18nOffersBlock.specifyDatesToCheckPrices()}
                            </div>
                        </MessageBoxPopup>
                    )}
                </Flex>
            );
        }

        const guestsCount = getGuestsCount(searchParams);

        if (!hasOffers) {
            return (
                <div
                    className={cx('emptyRoomOffer')}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'emptyOffer',
                    })}
                >
                    <div className={cx('emptyRoomOfferText')}>
                        {i18nBlock.emptyRoomOffersNew({guestsCount})}
                    </div>
                </div>
            );
        }

        return (
            <MainOffers
                className={cx('offers')}
                deviceType={deviceType}
                offers={offers}
                partnerOffers={partnerOffers || []}
                operatorById={operatorById}
                isFullOffersTab
                onOfferSelect={onOfferSelect}
                nightsCount={nightsCount}
                isRoomGroup
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'mainOffers',
                })}
            />
        );
    }

    function renderRoomGallery(): ReactNode {
        if (images?.length) {
            return (
                <HotelPageRoomGallery
                    className={cx('roomGallery')}
                    deviceType={deviceType}
                    images={images}
                    onImageClick={handleImageClick}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'gallery',
                    })}
                />
            );
        }

        if (isDesktop) {
            return (
                <div className={cx('withoutImageGallery')}>
                    <HotelImageStub
                        className={cx('withoutImage')}
                        captionText={i18nBlock.noImages()}
                    />
                </div>
            );
        }

        return null;
    }

    function renderRoomInfo(): ReactNode {
        return (
            <div className={cx('info')}>
                <Heading
                    level={isMobile ? 3 : 2}
                    {...prepareQaAttributes({parent: rest, current: 'name'})}
                >
                    {name}
                </Heading>
                <HotelOfferBadges className={cx('badges')} badges={badges} />
                {renderBedGroupsAndSize()}
                {renderMainAmenities()}
                {renderDetailedInfo()}
                {renderOffers()}
            </div>
        );
    }

    function renderMainAmenities(): ReactNode {
        if (mainAmenities?.length) {
            return (
                <Amenities
                    className={cx('mainAmenities')}
                    amenities={mainAmenities}
                    viewType={isDesktop ? 'table' : 'inline'}
                    amenityTextSize="m"
                    amenityFontWeight="normal"
                    handleMoreInfoClick={handleWatchButtonClick}
                    amenityLimit={isMobile ? SHORT_AMENITY_LIMIT : undefined}
                    renderMoreButton={isMobile}
                    deviceType={deviceType}
                    itemsInRow={hotelsHideFeatures ? 8 : 3}
                    hideFeatures={hotelsHideFeatures}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'amenities',
                    })}
                />
            );
        }

        return null;
    }

    function renderDetailedInfo(): ReactNode {
        if (isMobile) {
            return null;
        }

        return (
            <HotelPageRoomDetailedInfoButton
                className={cx('detailedInfo')}
                room={room}
                handleMoreInfoClick={handleWatchButtonClick}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'detailedInfoButton',
                })}
            />
        );
    }

    function renderBedGroupsAndSize(): ReactNode {
        if (!area && !bedGroups) {
            return null;
        }

        return (
            <div
                className={cx('bedGroupsAndSize')}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'bedGroupsAndSize',
                })}
            >
                <RoomBedGroupsAndSize area={area} bedGroups={bedGroups} />
            </div>
        );
    }

    return (
        <Card
            key={id}
            className={cx('room', 'room_full', deviceMods('room', deviceType))}
            ref={roomRef}
            {...prepareQaAttributes(rest)}
        >
            <Anchor anchorId={`${HOTEL_ROOM_ANCHOR}_${id}`} />
            {canShowImages && renderRoomGallery()}
            {renderRoomInfo()}
            {renderModalImagesViewer()}
        </Card>
    );
};

export default Room;
