import {Component, ReactNode} from 'react';
import {connect} from 'react-redux';
import {withRouter, RouteComponentProps} from 'react-router-dom';
import _flow from 'lodash/flow';

import {ERedirectStatusCodes} from 'constants/redirectStatusCodes';
import {URLs} from 'constants/urls';

import {HotelSlugType} from 'types/hotels/hotel/IHotel';

import {StoreInterface} from 'reducers/storeTypes';
import {
    fetchLegacyHotelInfoActions,
    StartFetchLegacyHotelInfoType,
} from 'reducers/hotels/legacyHotelPage/actions';

import {getLegacyHotelInfo} from 'selectors/hotels/legacyHotel/getLegacyHotelInfo';

import {hotelsURLs} from 'projects/hotels/utilities/urls';

import Spinner from 'components/Spinner/Spinner';
import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import RedirectWithStatus from 'components/RedirectWithStatus/RedirectWithStatus';

import DeviceTypeContext from 'contexts/DeviceTypeContext';
import {serverFetchDataDispatcher} from 'contexts/ServerFetchDataContext';

import {fetchLegacyHotelInfoSSR} from 'server/redux/hotels/fetchLegacyHotelInfo';

/* Init Styles */
import cx from './LegacyHotelPage.scss';

/* Constants */
const HOTELS_PATH = URLs.hotels;

/* ReduxContainer Types */
interface ILegacyHotelPageStatePropsContainer {
    legacyHotelInfo: ReturnType<typeof getLegacyHotelInfo>;
}

interface ILegacyHotelPageDispatchContainer {
    fetchLegacyHotelInfo: StartFetchLegacyHotelInfoType;
}

export interface ILegacyHotelPageProps
    extends ILegacyHotelPageStatePropsContainer,
        ILegacyHotelPageDispatchContainer,
        RouteComponentProps {}

/* ReduxContainer mapHelpers */
const mapStateToProps = (
    state: StoreInterface,
): ILegacyHotelPageStatePropsContainer => ({
    legacyHotelInfo: getLegacyHotelInfo(state),
});

const mapDispatchToProps = {
    fetchLegacyHotelInfo: fetchLegacyHotelInfoActions.request,
};

class LegacyHotelPage extends Component<ILegacyHotelPageProps> {
    componentDidMount(): void {
        this.fetchHotelInfo();
    }

    componentDidUpdate(prevProps: ILegacyHotelPageProps): void {
        this.checkErrorHotelFetch();
        this.checkSuccessHotelFetch(prevProps);
    }

    /* Check helpers */

    private checkErrorHotelFetch(): void {
        const {
            legacyHotelInfo: {isError},
        } = this.props;

        if (isError) {
            this.goToIndexHotelsPage();
        }
    }

    private checkSuccessHotelFetch(prevProps: ILegacyHotelPageProps): void {
        const {legacyHotelInfo: prevLegacyHotelInfo} = prevProps;
        const {legacyHotelInfo} = this.props;

        if (!prevLegacyHotelInfo.data && legacyHotelInfo.data) {
            const {
                data: {hotelSlug},
            } = legacyHotelInfo;

            if (hotelSlug) {
                this.goToHotelPageBySlug(hotelSlug);
            } else {
                this.goToIndexHotelsPage();
            }
        }
    }

    /* Helpers */

    private getLegacyHotelId(): string | undefined {
        const {
            location: {pathname},
        } = this.props;
        const matchResult = pathname.match(/^\/hotel\/(\d+)/);

        if (matchResult) {
            return matchResult[1];
        }

        return undefined;
    }

    private getRedirectPageFromLegacyHotel(): string | undefined {
        if (__SERVER__) {
            const {legacyHotelInfo} = this.props;
            const hotelSlug = legacyHotelInfo?.data?.hotelSlug;

            if (hotelSlug) {
                return hotelsURLs.getHotelUrl({hotelSlug});
            }

            return HOTELS_PATH;
        }

        return undefined;
    }

    /* Actions */

    private fetchHotelInfo(): void {
        const {fetchLegacyHotelInfo} = this.props;
        const legacyHotelId = this.getLegacyHotelId();

        if (legacyHotelId) {
            fetchLegacyHotelInfo({id: legacyHotelId});
        } else {
            this.goToIndexHotelsPage();
        }
    }

    private goToIndexHotelsPage(): void {
        const {history} = this.props;

        history.push(HOTELS_PATH);
    }

    private goToHotelPageBySlug(hotelSlug: HotelSlugType): void {
        const {history} = this.props;
        const hotelUrlWithSlug = hotelsURLs.getHotelUrl({hotelSlug});

        history.push(hotelUrlWithSlug);
    }

    /* Render */

    private renderLogo(): ReactNode {
        return (
            <div className={cx('logo')}>
                <YandexTravelLogo />
            </div>
        );
    }

    private renderPageLoader(): ReactNode {
        const {
            legacyHotelInfo: {isError, isLoading, data},
        } = this.props;
        const canRenderLoader = isLoading || isError || !data;

        if (canRenderLoader) {
            return (
                <div className={cx('loader')}>
                    <Spinner size="m" progress />
                </div>
            );
        }

        return null;
    }

    render(): ReactNode {
        const redirectPageFromLegacyHotel =
            this.getRedirectPageFromLegacyHotel();

        if (redirectPageFromLegacyHotel) {
            return (
                <RedirectWithStatus
                    to={redirectPageFromLegacyHotel}
                    statusCode={ERedirectStatusCodes.PERMANENTLY}
                />
            );
        }

        return (
            <DeviceTypeContext.Consumer>
                {({isMobile}): ReactNode => (
                    <div
                        className={cx('legacyHotelPage', {
                            legacyHotelPage_mobile: isMobile,
                        })}
                    >
                        {this.renderLogo()}
                        {this.renderPageLoader()}
                    </div>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

export default _flow(
    withRouter,
    connect<
        ILegacyHotelPageStatePropsContainer,
        ILegacyHotelPageDispatchContainer,
        {},
        StoreInterface
    >(mapStateToProps, mapDispatchToProps),
    serverFetchDataDispatcher([fetchLegacyHotelInfoSSR]),
)(LegacyHotelPage);
