import {PureComponent, ReactNode} from 'react';

import {URLs} from 'constants/urls';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {MapBoundsType} from 'types/common/ICoordinates';
import {NavigationTokenType} from 'types/hotels/search/INavigationTokens';
import {ReactRouterLocationType} from 'types/common/IReactRouterLocation';
import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';
import {EToggler} from 'types/common/togglers/EToggler';
import {IWithDeviceType} from 'types/withDeviceType';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {
    getSearchPageLayoutViewByQuery,
    getSearchPageLayoutViewQueryParams,
} from 'projects/hotels/utilities/getSearchPageLayoutViewParams/getSearchPageLayoutViewParams';
import updateLocationWithQuery from 'utilities/updateLocationWithQuery/updateLocationWithQuery';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import debounceById from 'utilities/functions/debounceById';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {loadable} from 'utilities/componentLoadable';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import SearchForm from 'projects/hotels/components/SearchForm/SearchForm';
import HotelsSearchController from 'projects/hotels/pages/SearchPage/components/HotelsSearchController/HotelsSearchController';
import TaxiDisclaimer from 'components/TaxiDisclaimer/TaxiDisclaimer';
import MirCashbackBanner from 'projects/hotels/components/MirCashbackBanner/MirCashbackBanner';
import LinkButton from 'components/LinkButton/LinkButton';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import ChunksPreloader from 'components/ChunksPreloader/ChunksPreloader';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import SearchFormScroller from './components/SearchFormScroller/SearchFormScroller';
import SearchFilters from 'projects/hotels/pages/SearchPage/components/SearchFilters/SearchFilters';
import HotelsSort from 'projects/hotels/components/HotelsSort/HotelsSort';

import HotelsSearchInformationProvider from 'projects/hotels/containers/HotelsSearchInformationProvider/HotelsSearchInformationProvider';
import {ISearchPage} from 'projects/hotels/pages/SearchPage/SearchPage';

import HotelsList from '../HotelsList/HotelsList';
import NavigationButtons from '../NavigationButtons/NavigationButtons';
import HotelsSearchInformationSection from '../HotelsSearchInformationSection/HotelsSearchInformationSection';

import cx from './BaseSearchPage.scss';

const HotelsSearchMap = loadable(
    () => import('../HotelsSearchMap/HotelsSearchMap'),
);

const PRELOAD_CHUNKS = [
    HotelsSearchMap,
    loadable(
        () => import('projects/hotels/pages/HotelPage/HotelPageContainer'),
    ),
];
const SET_ON_FAVORITES_CLICK_DEBOUNCE = 2000;

interface IBaseSearchPageProps
    extends ISearchPage,
        IWithDeviceType,
        IWithQaAttributes {}

interface IBaseSearchPageState {
    activeView: SearchPageLayoutView;
}

class BaseSearchPage extends PureComponent<
    IBaseSearchPageProps,
    IBaseSearchPageState
> {
    static getInitialActiveViewByLocation(
        location: ReactRouterLocationType,
    ): SearchPageLayoutView {
        const queryByLocation = getQueryByLocation(location);

        return getSearchPageLayoutViewByQuery(queryByLocation);
    }

    debouncedSearchReactionRequest = debounceById((permalink: string) => {
        this.props.changeHotelsIsFavoriteActions({permalink});
    }, SET_ON_FAVORITES_CLICK_DEBOUNCE);

    hotelsMapLoaded: boolean;

    constructor(props: IBaseSearchPageProps) {
        super(props);

        const {location} = props;

        this.state = {
            activeView: BaseSearchPage.getInitialActiveViewByLocation(location),
        };

        this.hotelsMapLoaded = false;
    }

    /* Actions */
    toggleActiveView = (): void => {
        const {activeView} = this.state;
        const toggledActiveView =
            activeView === SearchPageLayoutView.LIST
                ? SearchPageLayoutView.MAP
                : SearchPageLayoutView.LIST;

        this.handleChangeActiveView(toggledActiveView);
    };

    syncActiveViewWithLocation = (activeView: SearchPageLayoutView): void => {
        const {location} = this.props;
        const searchPageLayoutViewQueryParams =
            getSearchPageLayoutViewQueryParams(activeView);

        updateLocationWithQuery(searchPageLayoutViewQueryParams, location, {
            replace: true,
        });
    };

    /* Handlers */

    handleChangeActiveView = (activeView: SearchPageLayoutView): void => {
        this.setState({activeView}, () => {
            this.syncActiveViewWithLocation(activeView);
        });

        reachGoal(EHotelsGoal.SEARCH_PAGE_LIST_OR_MAP_VIEW_CHANGE);
    };

    handleMapBoundsChange = (bounds: MapBoundsType): void => {
        const {setMapBounds} = this.props;

        if (
            bounds[0].lon !== bounds[1].lon &&
            bounds[0].lat !== bounds[1].lat
        ) {
            setMapBounds(bounds);

            reachGoal(EHotelsGoal.SEARCH_PAGE_MAP_BOUNDS_CHANGE);
        }
    };

    handleChangeSearchNavigationToken = (
        navigationToken: NavigationTokenType,
    ): void => {
        const {setNavigationToken} = this.props;

        setNavigationToken(navigationToken);

        reachGoal(EHotelsGoal.SEARCH_PAGE_LIST_PAGE_CHANGE);
    };

    handleClickMarker = (permalink: PermalinkType): void => {
        const {activeView} = this.state;
        const hasFullScreenMap = activeView === SearchPageLayoutView.MAP;
        const {
            clickMapMarker,
            selectMapMarker,
            deviceType: {isMobile},
        } = this.props;

        if (isMobile) {
            selectMapMarker(permalink);
        }

        clickMapMarker(permalink);

        reachGoal(
            hasFullScreenMap
                ? EHotelsGoal.SEARCH_PAGE_MAP_PIN_FULL_VIEW_CLICK
                : EHotelsGoal.SEARCH_PAGE_MAP_PIN_SHORT_VIEW_CLICK,
        );
    };

    handleResetSelectedCard = (): void => {
        const {selectMapMarker} = this.props;

        selectMapMarker(undefined);
    };

    handleApplyFilters = (): void => {
        const {applyFilters} = this.props;

        applyFilters();
    };

    handleLoadMap = (): void => {
        this.hotelsMapLoaded = true;
    };

    handleAddFavoriteHotelOnSearchClick = ({
        permalink,
        isFavorite,
    }: {
        permalink: PermalinkType;
        isFavorite: boolean;
    }): void => {
        const {removeFavoriteHotel, addFavoriteHotel, setSnackbarInfo} =
            this.props;

        if (isFavorite) {
            removeFavoriteHotel({permalink});
            setSnackbarInfo({
                lastAction: `${permalink}-remove`,
                hasCancellation: true,
                message: i18nFavoritesBlock.removeFavoriteHotel(),
                cancelAction: 'addFavoriteHotelOnSearch',
                cancelActionData: {requestParams: {permalink}},
                page: 'search',
            });
        } else {
            addFavoriteHotel({
                requestParams: {permalink},
                message: insertJSXIntoKey(i18nFavoritesBlock.addFavoriteHotel)({
                    favorites: (
                        <LinkButton href={URLs.favorites} theme="brand">
                            {i18nFavoritesBlock.favorites()}
                        </LinkButton>
                    ),
                }),
            });
        }

        this.debouncedSearchReactionRequest(permalink, permalink);
    };

    handleNoMoreFavoritesModalClose = (): void => {
        const {changeIsOpenForNoMoreFavoritesModal} = this.props;

        changeIsOpenForNoMoreFavoritesModal?.(false);
    };

    /* Render */

    renderPreloadChunksControl(): ReactNode {
        return (
            <ChunksPreloader
                preloadChunks={PRELOAD_CHUNKS}
                maxProportionTimeout={20000}
            />
        );
    }

    renderSearchInformation(): ReactNode {
        const {deviceType} = this.props;

        if (deviceType.isDesktop) {
            return null;
        }

        return (
            <HotelsSearchInformationSection
                className={cx('searchInformation')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'searchInformation',
                })}
            />
        );
    }

    renderSearchForm(): ReactNode {
        return (
            <SearchFormScroller>
                <SearchForm
                    className={cx('searchForm')}
                    searchFormClassName="hotelsSearchForm"
                />
            </SearchFormScroller>
        );
    }

    renderFilters(mapListNode?: ReactNode): ReactNode {
        const {
            changeFilterGroup,
            resetFilters,
            revertFilters,
            changePriceFilter,
            resetFilter,
            experiments,
        } = this.props;
        const {activeView} = this.state;

        return (
            <SearchFilters
                onChangePriceFilter={changePriceFilter}
                onChangeFilterGroup={changeFilterGroup}
                onResetFilters={resetFilters}
                onRevertFilters={revertFilters}
                onApplyFilters={this.handleApplyFilters}
                onResetFilter={resetFilter}
                withNestedSheetsExp={experiments.hotelsFiltersCurtain}
                withScrollableSheetExp={experiments.hotelsFiltersSheet}
                mapListRadioButtonNode={mapListNode}
                renderSort={
                    activeView === SearchPageLayoutView.LIST
                        ? this.renderSortInfo
                        : undefined
                }
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'filtersBar',
                })}
            />
        );
    }

    protected renderMirCashbackBanner(className: string): ReactNode {
        const {deviceType, hasSearchData, isMirCashbackExpanded, hasMirBanner} =
            this.props;
        const {activeView} = this.state;

        if (!hasSearchData) {
            return (
                <Skeleton
                    className={cx(
                        'mirCashbackSkeleton',
                        deviceMods('mirCashbackSkeleton', deviceType),
                        {mirCashbackSkeleton_expanded: isMirCashbackExpanded},
                        className,
                    )}
                    withAnimation
                />
            );
        }

        if (!hasMirBanner || activeView !== SearchPageLayoutView.LIST) {
            return null;
        }

        return (
            <MirCashbackBanner
                className={className}
                type={EToggler.MIR_POSSIBLE_CASHBACK_SEARCH}
            />
        );
    }

    protected renderAdFoxBanner(className?: string): ReactNode {
        return (
            <AdFoxBanner
                className={className}
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
                fixed
            />
        );
    }

    renderSortInfo = (): ReactNode =>
        this.state.activeView === SearchPageLayoutView.LIST ? (
            <HotelsSort />
        ) : null;

    renderSearchNavigationButtons(): ReactNode {
        return (
            <NavigationButtons
                className={cx('searchNavigationButtons')}
                onChangeNavigationToken={this.handleChangeSearchNavigationToken}
                {...prepareQaAttributes(this.props)}
            />
        );
    }

    renderHotelsMap = (
        mapClassName: string = '',
        mapStyle: {width?: string | number; height?: string | number} = {},
    ): ReactNode => {
        const {setActiveHotel, resetActiveHotel, changeCurrentGeoIdFilter} =
            this.props;

        const {activeView} = this.state;
        const {height, width} = mapStyle;

        return (
            <HotelsSearchMap
                className={mapClassName}
                mapHeight={height}
                mapWidth={width}
                isFullViewMap={activeView === SearchPageLayoutView.MAP}
                onFavoriteClick={this.handleAddFavoriteHotelOnSearchClick}
                onLoadMap={this.handleLoadMap}
                onBoundsChange={this.handleMapBoundsChange}
                onHoverMapMarker={setActiveHotel}
                onLeaveMapMarker={resetActiveHotel}
                onClickMapMarker={this.handleClickMarker}
                onResetSelectedCard={this.handleResetSelectedCard}
                onChangeCurrentGeoIdFilter={changeCurrentGeoIdFilter}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'map',
                })}
            />
        );
    };

    renderHotelsList(hotelsListClassName: string = ''): ReactNode {
        const {setActiveHotel, resetActiveHotel} = this.props;
        const {activeView} = this.state;
        const canHideHotelList = activeView !== SearchPageLayoutView.LIST;

        return (
            <HotelsList
                className={cx(
                    {
                        hotelsList_hide: canHideHotelList,
                    },
                    hotelsListClassName,
                )}
                onHoverHotelCard={setActiveHotel}
                onLeaveHotelCard={resetActiveHotel}
                onFavoriteClick={this.handleAddFavoriteHotelOnSearchClick}
                onCloseModalClick={this.handleNoMoreFavoritesModalClose}
                {...prepareQaAttributes(this.props)}
            />
        );
    }

    renderHotelsSearchController(): ReactNode {
        const {stopSearch, startSearch, updateCountFilters, clearSearchResult} =
            this.props;
        const {activeView} = this.state;

        return (
            <HotelsSearchController
                activeView={activeView}
                updateCountFilters={updateCountFilters}
                startSearch={startSearch}
                stopSearch={stopSearch}
                clearSearchResult={clearSearchResult}
            />
        );
    }

    renderSearchInformationProvider(): ReactNode {
        return <HotelsSearchInformationProvider />;
    }

    renderPromoDisclaimer(): ReactNode {
        if (!this.props.hasPromoTaxi) {
            return null;
        }

        return <TaxiDisclaimer />;
    }
}

export default BaseSearchPage;
