import React, {ReactNode, CSSProperties, useCallback, useEffect} from 'react';
import {useSelector} from 'react-redux';

import {
    IChangeFilterGroupPayload,
    IChangeFilterGroupPayloadWithTarget,
} from 'types/hotels/search/IFiltersInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {searchMapSelector} from 'projects/hotels/pages/SearchPage/components/DesktopSearchPage/components/SearchPageMap/selectors/searchMapSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {params, reachGoal} from 'utilities/metrika';

import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import HotelsQuickFilters from 'projects/hotels/components/HotelsFilters/components/HotelsQuickFilters/HotelsQuickFilters';
import Flex from 'components/Flex/Flex';
import HotelsFiltersBar from 'projects/hotels/components/HotelsFilters/components/HotelsFiltersBar/HotelsFiltersBar';

import ToggleMapViewButton from '../ToggleMapViewButton/ToggleMapViewButton';

import cx from './SearchPageMap.scss';

export interface ISearchPageMapProps {
    windowHeight: number;
    hasFullScreenMap: boolean;

    renderHotelsMap(className: string, style: CSSProperties): ReactNode;
    onToggleMapView(): void;
    onChangeFilterGroup?(payload: IChangeFilterGroupPayloadWithTarget): void;
}

const SearchPageMap: React.FC<ISearchPageMapProps> = props => {
    const {
        hasFullScreenMap,
        renderHotelsMap,
        onToggleMapView,
        onChangeFilterGroup,
    } = props;

    const deviceType = useDeviceType();

    useEffect(() => {
        if (deviceType.isDesktop && hasFullScreenMap) {
            params({
                hotels: {searchPage: {didFullscreenDesktopMapView: 1}},
            });
        }
    }, [deviceType.isDesktop, hasFullScreenMap]);

    const showFiltersBar = hasFullScreenMap && deviceType.isDesktop;

    const handleChangeQuickFilters = useCallback(
        (payload: IChangeFilterGroupPayload) => {
            if (!onChangeFilterGroup) return;

            onChangeFilterGroup({
                ...payload,
                targetFiltersType: 'ALL',
            });

            reachGoal(EHotelsGoal.SEARCH_PAGE_QUICK_FILTERS_CHANGE);
        },
        [onChangeFilterGroup],
    );

    const {canRenderMap} = useSelector(searchMapSelector);

    if (!canRenderMap) {
        return <Skeleton className={cx('mapSkeleton')} withAnimation />;
    }

    return (
        <>
            <Flex between="2" inline className={cx('controls')}>
                <ToggleMapViewButton
                    className={cx('toggleViewButton')}
                    canToggleToFullView={!hasFullScreenMap}
                    onClick={onToggleMapView}
                />

                {showFiltersBar && (
                    <HotelsFiltersBar
                        className={cx('filtersBar')}
                        deviceInfo={deviceType}
                        quickFiltersNode={
                            <HotelsQuickFilters
                                // TODO Удалить после отказа от quickFilters
                                theme="raised"
                                dirtyPatchDesktopControls
                                className={cx('mapHotelsQuickFilters')}
                                onChange={handleChangeQuickFilters}
                            />
                        }
                    />
                )}
            </Flex>

            {renderHotelsMap(cx('map'), {
                width: hasFullScreenMap ? '100%' : undefined,
            })}
        </>
    );
};

export default React.memo(SearchPageMap);
