import React, {useCallback, useEffect, useRef} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {SEARCH_TTL} from 'projects/hotels/constants/searchPage';

import {EStartSearchReason} from 'types/hotels/search/ISearchReason';
import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';

import {
    ClearSearchResultType,
    StartSearchHotelsType,
    StopSearchType,
} from 'reducers/hotels/searchPage/search/actions';
import {UpdateCountFiltersActionType} from 'reducers/hotels/searchPage/filters/actions';
import {fetchWelcomePromocode} from 'reducers/hotels/welcomePromocode/thunk';

import {getWelcomePromocodeFullInfo} from 'selectors/hotels/welcomePromocode/getWelcomePromocode';
import {searchControllerSelector} from 'projects/hotels/pages/SearchPage/components/HotelsSearchController/selectors/searchControllerSelector';

import {convertBoundsToString} from 'components/YandexMaps/utilities';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {isExpired} from './utilities/isExpired';
import {checkDifferenceBetweenQueryAndSearchInfoParams} from 'projects/hotels/pages/SearchPage/components/HotelsSearchController/utilities/checkDifferenceBetweenQueryAndSearchInfoParams';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

import {useLastSearchTimeMarkerChange} from 'projects/hotels/hooks/useLastSearchTimeMarkerChange';

export interface IHotelsSearchControllerProps {
    activeView: SearchPageLayoutView;

    startSearch: StartSearchHotelsType;
    stopSearch: StopSearchType;
    clearSearchResult: ClearSearchResultType;
    updateCountFilters: UpdateCountFiltersActionType;
}

const HotelsSearchController: React.FC<IHotelsSearchControllerProps> =
    props => {
        const {stopSearch, clearSearchResult, startSearch, updateCountFilters} =
            props;

        const {
            currentFilters,
            permanentFilters,
            offerSearchParams,
            needSyncCurrentFiltersWithServer,
            needSyncPermanentFiltersWithServer,
            needSyncSortWithServer,
            navigationTokens,
            navigationToken,
            mapBounds,
            mapBoundsBySearch,
            isSearchFinished,
            geoId,
            lastSearchTimestamp,
            sortOrigin,
            topHotelSlug,
        } = useSelector(searchControllerSelector);

        const history = useHistory();
        const deviceType = useDeviceType();
        const dispatch = useDispatch();
        const welcomePromocodeData = useSelector(getWelcomePromocodeFullInfo);
        const isSearchStarted = useRef(false);

        const {location} = history;

        isSearchStarted.current = false;

        const searchOnce: StartSearchHotelsType = useCallback(
            payload => {
                if (!isSearchStarted.current) {
                    isSearchStarted.current = true;
                    startSearch(payload);
                }
            },
            [startSearch],
        );

        /* 1. Mount */
        useEffect(() => {
            if (!welcomePromocodeData.isSet) {
                dispatch(fetchWelcomePromocode({}));
            }

            if (deviceType.isMobile) {
                const isSearchExpired = isExpired(
                    lastSearchTimestamp,
                    SEARCH_TTL,
                );

                if (isSearchExpired || !isSearchFinished) {
                    searchOnce({
                        startSearchReason: EStartSearchReason.MOUNT,
                    });
                }
            } else {
                searchOnce({
                    startSearchReason: EStartSearchReason.MOUNT,
                });
            }
        }, [clearSearchResult, history, searchOnce]);

        /* 2. Location changed */
        const changedLastSearchTimeMarker =
            useLastSearchTimeMarkerChange(location);

        useEffect(() => {
            if (geoId && offerSearchParams) {
                const canStartSearchByQuery =
                    checkDifferenceBetweenQueryAndSearchInfoParams(location, {
                        geoId,
                        navigationToken,
                        offerSearchParams,
                        filters: permanentFilters,
                        mapBounds: mapBoundsBySearch,
                        sortOrigin,
                        topHotelSlug,
                    });

                const {geoId: queryGeoId} = getQueryByLocation(location);

                const hasGeoChanged =
                    geoId && queryGeoId && String(geoId) !== queryGeoId;

                if (canStartSearchByQuery || changedLastSearchTimeMarker) {
                    const reason = hasGeoChanged
                        ? EStartSearchReason.QUERY_BY_LOCATION
                        : EStartSearchReason.QUERY_WITH_SAME_GEO;

                    searchOnce({
                        startSearchReason: reason,
                    });
                }
            }
        }, [location.search, searchOnce]);

        /* 3. MapBounds changed */
        useEffect(() => {
            if (
                mapBounds &&
                mapBoundsBySearch &&
                mapBounds !== mapBoundsBySearch &&
                convertBoundsToString(mapBoundsBySearch) !==
                    convertBoundsToString(mapBounds)
            ) {
                searchOnce({startSearchReason: EStartSearchReason.MAP_BOUNDS});
            }
        }, [mapBounds, mapBoundsBySearch, searchOnce]);

        /* 4. NavigationToken changed */
        useEffect(() => {
            if (
                navigationToken &&
                navigationTokens &&
                navigationTokens.currentPage !== navigationToken
            ) {
                searchOnce({
                    startSearchReason: EStartSearchReason.NAVIGATION_TOKEN,
                });
            }
        }, [navigationToken, navigationTokens, searchOnce]);

        /* 5. PermanentFilters changed */
        useEffect(() => {
            if (needSyncPermanentFiltersWithServer) {
                searchOnce({
                    startSearchReason: EStartSearchReason.FILTERS,
                });
            }
        }, [
            permanentFilters,
            needSyncPermanentFiltersWithServer,
            searchOnce,
            isSearchStarted,
        ]);

        /* 6. CurrentFilters changed */
        useEffect(() => {
            if (needSyncCurrentFiltersWithServer) {
                updateCountFilters();
            }
        }, [
            currentFilters,
            needSyncCurrentFiltersWithServer,
            updateCountFilters,
        ]);

        /* 7. Sort changed */
        useEffect(() => {
            if (needSyncSortWithServer) {
                searchOnce({
                    startSearchReason: EStartSearchReason.SORT,
                });
            }
        }, [needSyncSortWithServer, searchOnce]);

        /* 8. WillUnMount */
        useEffect(() => {
            return (): void => {
                stopSearch();

                if (deviceType.isDesktop) {
                    clearSearchResult();
                }
            };
        }, [stopSearch, clearSearchResult]);

        return null;
    };

export default React.memo(HotelsSearchController);
