import React from 'react';

import {IHotelWithOffers} from 'types/hotels/hotel/IHotelWithOffers';
import {EPortalSeedReferer} from 'types/hotels/common/IAttribution';
import {IHotelWithMinPrice} from 'types/hotels/hotel/IHotelWithMinPrice';
import {IWithClassName} from 'types/withClassName';

import {
    calculateCardPosition,
    IMarkerPosition,
} from 'projects/hotels/pages/SearchPage/components/HotelsSearchMap/utilities/calculateMapCardPosition';
import {ISize, useElementSize} from 'utilities/hooks/useElementSize';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HotelShortView from 'projects/hotels/components/HotelShortView/HotelShortView';
import Card from 'components/Card/Card';
import PlusLabel from 'components/PlusLabel/PlusLabel';

import cx from './MapHotelCard.scss';

interface IMapHotelCardProps extends IWithClassName, IWithQaAttributes {
    markerPosition: IMarkerPosition;
    containerSize: ISize;
    hotel: IHotelWithOffers | IHotelWithMinPrice;
    hotelsPercentDiscount?: boolean;
}

const MapHotelCard: React.FC<IMapHotelCardProps> = props => {
    const {
        className,
        markerPosition,
        containerSize,
        hotel,
        hotelsPercentDiscount,
    } = props;

    const permalink = hotel.hotel.permalink;

    const {ref: cardRef, size: cardSize} = useElementSize<
        string,
        HTMLDivElement
    >({id: permalink});
    const {top, left} = calculateCardPosition({
        markerPosition,
        containerSize,
        cardSize,
    });

    const yandexPlusInfo =
        'offers' in hotel && hotel.offers?.[0].offerYandexPlusInfo;

    return (
        <Card
            className={cx('hotel', className)}
            ref={cardRef}
            style={{
                top,
                left,
            }}
            inset={2}
            shadow="popup"
            radius="s"
            {...prepareQaAttributes(props)}
        >
            <HotelShortView
                seed={EPortalSeedReferer.PORTAL_HOTELS_SEARCH}
                hotel={hotel.hotel}
                offers={'offers' in hotel ? hotel.offers : undefined}
                canRenderOfferPrice
                hotelsPercentDiscount={hotelsPercentDiscount}
                minPrice={'minPrice' in hotel ? hotel.minPrice : undefined}
            />

            {yandexPlusInfo && (
                <PlusLabel
                    className={cx('plusPoints')}
                    count={yandexPlusInfo.points}
                />
            )}
        </Card>
    );
};

export default MapHotelCard;
