import {ReactNode} from 'react';

import {SearchPageLayoutView} from 'types/hotels/search/ISearchLayout';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import scrollTo from 'utilities/dom/scrollTo';

import MobileSearchPageLayout from 'projects/hotels/pages/SearchPage/components/MobileSearchPageLayout/MobileSearchPageLayout';
import ForceLogin from 'projects/hotels/components/ForceLogin/ForceLogin';
import HangingYandexMessenger from 'components/YandexMessenger/components/HangingYandexMessenger/HangingYandexMessenger';
import SearchPageHeader from './components/SearchPageHeader/SearchPageHeader';

import BaseSearchPage from '../BaseSearchPage/BaseSearchPage';
import InfiniteScroll from '../InfiniteScroll/InfiniteScroll';
import HangingActiveViewButton from '../HangingActiveViewButton/HangingActiveViewButton';

import cx from './MobileSearchPage.scss';

const FIXED_FILTERS_BAR_HEIGHT = 52;

// @link https://st.yandex-team.ru/HOTELS-5945
// В рамках задачи попросили убрать баннер, но есть вероятность,
// что он еще вернется, поэтому просто выключил его флагом в коде
const ENABLE_ADFOX_BANNER = false;

class MobileSearchPage extends BaseSearchPage {
    componentDidMount(): void {
        const {activeView} = this.state;

        if (activeView === SearchPageLayoutView.MAP) {
            scrollTo({top: 0});
        }
    }

    /* Hotels List */

    private renderMobileBanners(): ReactNode {
        const {activeView} = this.state;
        const {hasMirBanner} = this.props;

        if (hasMirBanner) {
            return this.renderMirCashbackBanner(cx('banner'));
        }

        const isHotelListView = activeView === SearchPageLayoutView.LIST;

        if (!ENABLE_ADFOX_BANNER || !isHotelListView) {
            return null;
        }

        return this.renderAdFoxBanner(cx('banner'));
    }

    private renderHotelsListContainer(): ReactNode {
        const {activeView} = this.state;
        const isHotelListView = activeView === SearchPageLayoutView.LIST;

        return (
            <div
                className={cx('hotelsListContainer', {
                    hotelsListContainer_withMap: !isHotelListView,
                })}
            >
                {this.renderMobileBanners()}
                <ForceLogin className={cx('forceLogin')} isSearchPage />
                {this.renderHotelsList(cx('hotelsList'))}
                {this.renderNavigation()}
            </div>
        );
    }

    private renderNavigation(): ReactNode {
        const {activeView} = this.state;

        if (activeView === SearchPageLayoutView.MAP) {
            return null;
        }

        return <InfiniteScroll />;
    }

    /* Map */

    private renderHotelsMapContainer(): ReactNode {
        const {activeView} = this.state;
        const {windowWidth, windowHeight} = this.props;
        const mapHeight = windowHeight - FIXED_FILTERS_BAR_HEIGHT;

        if (activeView !== SearchPageLayoutView.MAP) {
            return null;
        }

        return (
            <div className={cx('mapContainer')}>
                {Boolean(mapHeight || this.hotelsMapLoaded) &&
                    this.renderHotelsMap(cx('map'), {
                        height: mapHeight,
                        width: windowWidth,
                    })}
            </div>
        );
    }

    /* Filters */

    private renderHangingButtons() {
        const {activeView} = this.state;

        return (
            <HangingActiveViewButton
                activeView={activeView}
                onClick={this.toggleActiveView}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'mapListRadioButton',
                })}
            />
        );
    }

    private renderFiltersAndSort(): ReactNode {
        const {activeView} = this.state;

        if (activeView === SearchPageLayoutView.MAP) {
            return this.renderFilters();
        }

        return (
            <SearchPageHeader className={cx('searchPageHeader')}>
                {this.renderFilters()}
            </SearchPageHeader>
        );
    }

    render(): ReactNode {
        const {activeView} = this.state;
        const isListView = activeView === SearchPageLayoutView.LIST;
        const isMapView = activeView === SearchPageLayoutView.MAP;

        return (
            <MobileSearchPageLayout
                className={cx(
                    'mobileSearchPage',
                    isMapView && 'mobileSearchPage_map',
                )}
                searchFormCollapsedClassName={cx('searchForm')}
                canShowHeader={isListView}
                canShowFooter={isListView}
                // в редизайне SearchInformation рендерится в SearchPageHeader
                hideSearchInformation={isListView}
            >
                {this.renderPreloadChunksControl()}
                {this.renderHotelsSearchController()}
                {this.renderSearchInformationProvider()}
                {this.renderFiltersAndSort()}
                {this.renderHotelsMapContainer()}
                {this.renderHotelsListContainer()}
                <div className={cx('disclaimer')}>
                    {this.renderPromoDisclaimer()}
                </div>
                {isListView && (
                    <HangingYandexMessenger
                        entrypoint="hotelsSearch"
                        metrikaGoal={EHotelsGoal.HOTELS_CHAT_CLICK}
                    />
                )}
                {this.renderHangingButtons()}
            </MobileSearchPageLayout>
        );
    }
}

export default MobileSearchPage;
