import {FC, useCallback, useEffect} from 'react';
import {useSelector} from 'react-redux';

import {ECommonGoal} from 'utilities/metrika/types/goals/common';
import {IWithClassName} from 'types/withClassName';

import {getSearchInformation} from 'selectors/hotels/search/searchInformation/searchInformationSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {reachGoal} from 'utilities/metrika';
import {useScrollHiding} from 'components/Hanging/utilities/useScrollHiding';

import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import Modal, {EModalAnimationType} from 'components/Modal/Modal';
import HotelsSearchForm from 'projects/hotels/components/SearchForm/SearchForm';

import HotelsSearchInformation from 'projects/hotels/containers/HotelsSearchInformation/HotelsSearchInformation';

import cx from './SearchPageHeader.scss';

const HEADER_HEIGHT = 52;
const SEARCH_INFO_HEIGHT = 120;

interface ISearchPageHeaderProps extends IWithClassName {}

const SearchPageHeader: FC<ISearchPageHeaderProps> = ({
    children,
    className,
}) => {
    const {geoObject} = useSelector(getSearchInformation);
    const deviceType = useDeviceType();
    const {
        value: isSearchFormVisible,
        setTrue: showSearchForm,
        setFalse: hideSearchForm,
    } = useBoolean(false);
    const {
        value: withShadow,
        setTrue: showShadow,
        setFalse: hideShadow,
    } = useBoolean(false);
    const isHidden = useScrollHiding({
        visibleArea: SEARCH_INFO_HEIGHT + HEADER_HEIGHT,
    });
    const isReady = Boolean(geoObject?.name);

    const handleSearchInformationClick = useCallback(() => {
        showSearchForm();

        reachGoal(ECommonGoal.HEADER_SEARCH_FORM_INFORMATION_CLICK);
    }, [showSearchForm]);

    const handleScroll = useCallback(() => {
        if (window.scrollY < HEADER_HEIGHT) {
            hideShadow();
        } else {
            showShadow();
        }
    }, [showShadow, hideShadow]);

    useEffect(() => {
        window.addEventListener('scroll', handleScroll);

        return (): void => {
            window.removeEventListener('scroll', handleScroll);
        };
    }, [handleScroll]);

    return (
        <>
            <div
                className={cx(
                    'root',
                    isHidden && 'root_hidden',
                    withShadow && 'root_withShadow',
                    className,
                )}
            >
                <div className={cx('searchInfo')}>
                    {isReady ? (
                        <HotelsSearchInformation
                            onClick={handleSearchInformationClick}
                            deviceType={deviceType}
                        />
                    ) : (
                        <Skeleton className={cx('skeleton')} />
                    )}
                </div>
                {children}
            </div>
            <ModalWithHistoryBack
                isMobile
                fullScreen
                disableAutoFocus
                isVisible={isSearchFormVisible}
                verticalAlign="top"
                autoclosable={false}
                animation={EModalAnimationType.NONE}
                onClose={hideSearchForm}
            >
                <Modal.Content>
                    <HotelsSearchForm
                        className={cx('searchForm')}
                        onSubmit={hideSearchForm}
                    />
                </Modal.Content>
            </ModalWithHistoryBack>
        </>
    );
};

export default SearchPageHeader;
