import React, {FC, ReactNode, useCallback} from 'react';
import {useSelector} from 'react-redux';

import {
    IChangeFilterGroupMeta,
    IChangeFilterGroupPayload,
    IChangeFilterGroupPayloadWithTarget,
    IChangePriceFilterPayloadWithTarget,
} from 'types/hotels/search/IFiltersInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IResetFilterAction} from 'server/api/HotelsSearchAPI/types/IResetFilterInfo';

import {hotelsFiltersSelector} from 'projects/hotels/pages/SearchPage/components/SearchFilters/selectors/hotelsFiltersSelector';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HotelsFilters from 'projects/hotels/components/HotelsFilters/HotelsFilters';

interface ISearchFiltersProps extends IWithQaAttributes {
    mapListRadioButtonNode?: React.ReactNode;

    onChangeFilterGroup(payload: IChangeFilterGroupPayloadWithTarget): void;
    onChangePriceFilter(payload: IChangePriceFilterPayloadWithTarget): void;
    onResetFilters(): void;
    onRevertFilters(): void;
    onResetFilter(payload: IResetFilterAction): void;
    onApplyFilters(): void;
    renderSort?: () => ReactNode;

    withNestedSheetsExp?: boolean;
    withScrollableSheetExp?: boolean;
}

const SearchFilters: FC<ISearchFiltersProps> = props => {
    const deviceType = useDeviceType();
    const {
        mapListRadioButtonNode,
        onApplyFilters,
        onRevertFilters,
        onResetFilters,
        onChangeFilterGroup,
        onChangePriceFilter,
        renderSort,
        onResetFilter,
        withNestedSheetsExp,
        withScrollableSheetExp,
    } = props;
    const {
        nights,
        filters,
        experiments,
        canDisableFilters,
        totalActiveFilters,
    } = useSelector(hotelsFiltersSelector);

    const handleApplyFilters = useCallback(() => {
        onApplyFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_APPLY);
    }, [onApplyFilters]);

    const handleResetFilters = useCallback(() => {
        onResetFilters();
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_RESET);
    }, [onResetFilters]);

    const handleChangeFilterGroup = useCallback(
        (
            payload: IChangeFilterGroupPayloadWithTarget,
            meta: IChangeFilterGroupMeta,
        ) => {
            onChangeFilterGroup(payload);

            const filterChanges = {
                [meta.filterId]: meta.itemId,
            };

            reachGoal(EHotelsGoal.SEARCH_PAGE_FILTER_GROUP_CHANGE, {
                hotels: {
                    filters:
                        payload.atomsOn?.length || meta.filterId === 'ratings'
                            ? {apply: filterChanges}
                            : {unset: filterChanges},
                },
            });
        },
        [onChangeFilterGroup],
    );

    const handleChangeQuickFilters = useCallback(
        (payload: IChangeFilterGroupPayload) => {
            onChangeFilterGroup({
                ...payload,
                targetFiltersType: 'ALL',
            });

            reachGoal(EHotelsGoal.SEARCH_PAGE_QUICK_FILTERS_CHANGE);
        },
        [onChangeFilterGroup],
    );

    const handleChangePriceFilter = useCallback(
        (payload: IChangePriceFilterPayloadWithTarget) => {
            onChangePriceFilter(payload);
            reachGoal(EHotelsGoal.SEARCH_PAGE_PRICE_FILTER_CHANGE);
        },
        [onChangePriceFilter],
    );

    const handleBaseCloseFilters = useCallback(() => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_FILTERS_CROSS_CLICK);
    }, []);

    const handleToggleViewFilters = useCallback(() => {
        reachGoal(EHotelsGoal.SEARCH_PAGE_ALL_FILTERS_BUTTON_CLICK);
    }, []);

    return (
        <HotelsFilters
            nights={nights}
            filters={filters}
            renderSort={renderSort}
            deviceInfo={deviceType}
            experiments={experiments}
            canDisableFilters={canDisableFilters}
            totalActiveFilters={totalActiveFilters}
            mapListRadioButtonNode={mapListRadioButtonNode}
            onChangeFilterGroup={handleChangeFilterGroup}
            onChangePriceFilter={handleChangePriceFilter}
            onResetFilters={handleResetFilters}
            onResetFilter={onResetFilter}
            onRevertFilters={onRevertFilters}
            onApplyFilters={handleApplyFilters}
            onChangeQuickFilter={handleChangeQuickFilters}
            onBaseCloseFilters={handleBaseCloseFilters}
            onToggleViewFilters={handleToggleViewFilters}
            withNestedSheetsExp={withNestedSheetsExp}
            withScrollableSheetExp={withScrollableSheetExp}
            {...prepareQaAttributes(props)}
        />
    );
};

export default SearchFilters;
