import React, {useCallback, useEffect, useRef, useState} from 'react';

import {
    ETravellineOfferMessageType,
    ITravellineOfferMessage,
} from 'types/hotels/legalOffer/ITravellineOfferMessage';

import {useToggle} from 'utilities/hooks/useToggle';
import {useBoolean} from 'utilities/hooks/useBoolean';
import checkOrigin from 'projects/hotels/pages/TravellineOfferPage/utilities/checkOrigin';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import Spinner from 'components/Spinner/Spinner';
import OfferForm from 'projects/hotels/pages/TravellineOfferPage/components/OfferForm/OfferForm';

import cx from './TravellineOfferPage.scss';

const QA_PREFIX = 'hotels-travellineOfferPage';

const TravellineOfferPage: React.FC = () => {
    const [isChecked, toggleChecked] = useToggle(false);
    const [savedToken, setSavedToken] = useState<string>();
    const messageSource = useRef<WindowProxy | null>(null);
    const [messageOrigin, setMessageOrigin] = useState<string>();
    const {value: loaderShown, setTrue: showLoader} = useBoolean(false);

    const buttonDisabled = !isChecked || !savedToken;

    const handleWindowMessage = useCallback((e: MessageEvent) => {
        const {source, origin} = e;

        if (!checkOrigin(origin)) {
            return;
        }

        const {type, payload} = e.data as ITravellineOfferMessage;

        switch (type) {
            case ETravellineOfferMessageType.SET_TOKEN:
                messageSource.current = source as WindowProxy;

                if (payload && payload.token) {
                    setSavedToken(payload.token);
                    setMessageOrigin(origin);
                }

                break;
            case ETravellineOfferMessageType.CLOSE_POPUP:
                window.close();

                break;
        }
    }, []);

    const notifyPageLoaded = useCallback(() => {
        const opener: WindowProxy | undefined = window.opener;

        if (!opener) {
            return;
        }

        const message: ITravellineOfferMessage = {
            type: ETravellineOfferMessageType.PAGE_LOADED,
        };

        opener.postMessage(message, '*');
    }, []);

    useEffect(() => {
        window.addEventListener('message', handleWindowMessage);
        notifyPageLoaded();

        return (): void => {
            window.removeEventListener('message', handleWindowMessage);
        };
    }, [handleWindowMessage, notifyPageLoaded]);

    const handleOfferAccept = useCallback(() => {
        if (
            !messageSource.current ||
            !savedToken ||
            !isChecked ||
            !messageOrigin
        ) {
            return;
        }

        const message: ITravellineOfferMessage = {
            type: ETravellineOfferMessageType.OFFER_ACCEPTED,
            payload: {token: savedToken},
        };

        messageSource.current.postMessage(message, messageOrigin);
        showLoader();
    }, [isChecked, savedToken, showLoader, messageOrigin]);

    return (
        <div className={cx('root')}>
            <YandexTravelLogo
                {...prepareQaAttributes({parent: QA_PREFIX, current: 'logo'})}
            />
            {loaderShown ? (
                <div className={cx('spinner')}>
                    <Spinner size="xl" />
                </div>
            ) : (
                <OfferForm
                    isChecked={isChecked}
                    buttonDisabled={buttonDisabled}
                    toggleChecked={toggleChecked}
                    onButtonClick={handleOfferAccept}
                    {...prepareQaAttributes(QA_PREFIX)}
                />
            )}
        </div>
    );
};

export default TravellineOfferPage;
