import {
    IDebugOfferSearchParams,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {TFilterAtomType, IFilterParams} from 'types/hotels/search/IFiltersInfo';
import {GeoIdType} from 'types/hotels/hotel/IGeoRegion';
import {ISearchPageQueryParams} from 'types/hotels/common/IQueryParams';
import {ISortParams} from 'types/hotels/search/ISortInfo';
import {EGeoLocationStatus} from 'types/common/browserGeolocation';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import {validateSearchHotelsParams} from 'projects/hotels/utilities/validateSearchHotelsParams/validateSearchHotelsParams';
import {reachGoal} from 'utilities/metrika';

export interface ISearchParamsByLocation
    extends Partial<IRequiredOfferParams>,
        Partial<IDebugOfferSearchParams>,
        IFilterParams,
        Partial<ISortParams> {
    navigationToken?: string;
    geoId?: GeoIdType;
    bbox?: string;
    sortOrigin?: string;
    topHotelSlug?: HotelSlugType;
    geoLocationStatus?: EGeoLocationStatus;
}

const parseFilterAtoms = (
    filterAtoms?: string,
): TFilterAtomType[] | undefined => {
    if (filterAtoms) {
        return filterAtoms.split(',');
    }
};

const tryDecodeSearchParamFromLocation = (
    searchParam?: string,
): string | undefined => {
    if (!searchParam) {
        return undefined;
    }

    try {
        return decodeURIComponent(searchParam);
    } catch (e) {
        return undefined;
    }
};

export const getSearchParamsByLocation = ():
    | ISearchParamsByLocation
    | undefined => {
    const queryByLocation = getQueryByBrowserHistory();

    if (queryByLocation) {
        const {
            geoId,
            adults,
            childrenAges,
            checkinDate,
            checkoutDate,
            onlyCurrentGeoId,
        } = validateSearchHotelsParams(queryByLocation);

        const {
            bbox,
            filterAtoms,
            searchText,
            maxPrice,
            minPrice,
            navigationToken,
            selectedSortId,
            sortOrigin,
            topHotelSlug,
        } = queryByLocation as ISearchPageQueryParams;

        if (searchText) {
            reachGoal(EHotelsGoal.HOTELS_ATTEMPT_TO_SEARCH_BY_TEXT);
        }

        return {
            geoId,
            adults,
            childrenAges,
            checkinDate,
            checkoutDate,
            bbox: tryDecodeSearchParamFromLocation(bbox),
            filterPriceFrom: minPrice,
            filterPriceTo: maxPrice,
            navigationToken,
            selectedSortId,
            onlyCurrentGeoId,
            filterAtoms: parseFilterAtoms(filterAtoms),
            sortOrigin,
            topHotelSlug,
            geoLocationStatus: sortOrigin
                ? EGeoLocationStatus.AVAILABLE
                : EGeoLocationStatus.UNKNOWN,
        };
    }
};
