import {URLs} from 'constants/urls';

import {IBaseSearchQueryParams} from 'types/hotels/common/IQueryParams';
import {HotelSlugType, PermalinkType} from 'types/hotels/hotel/IHotel';
import {ESearchSuggestItemType} from 'types/hotels/common/ISearchSuggest';
import {IHotelsSearchFormValues} from 'projects/hotels/components/SearchForm/types';

import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import {
    getHotelOfferToken,
    getOfferAttributionParams,
} from 'projects/hotels/utilities/getHotelPageParams/getOfferAttributionParamsByLocation';
import {getViewModePageParamsByQuery} from 'projects/hotels/utilities/getHotelPageParams/getHotelPageParams';
import {internalUrl} from 'utilities/url';
import getISObyDate from 'components/Calendar/utilities/getISObyDate';
import {getAttributionQueryParams} from 'projects/hotels/utilities/getAttributionParams/getAttributionParams';
import {createLastSearchTimeMarker} from 'projects/hotels/utilities/getLastSearchTimeMarkerQuery/getLastSearchTimeMarkerQuery';
import {getSearchPageLayoutViewQueryParams} from 'projects/hotels/utilities/getSearchPageLayoutViewParams/getSearchPageLayoutViewParams';
import {hotelsURLs} from 'projects/hotels/utilities/urls/index';
import {getSearchPageFilterQueryParamsByLocation} from 'projects/hotels/utilities/getSearchPageFilterQueryParamsByLocation/getSearchPageFilterQueryParamsByLocation';
import {getStoredSrcQueryParams} from 'utilities/srcParams/srcParams';
import {getSearchPageTopHotelParamByLocation} from 'projects/hotels/utilities/getSearchPageTopHotelSlugParamByLocation/getSearchPageTopHotelParamByLocation';

interface ISearchLinkHotelParams {
    hotelPermalink?: PermalinkType;
}

export interface ISearchLinkOptions {
    hasProductionOffers: boolean;
    hasDirectionSuggest?: boolean;
}

const getHotelSlugByFields = (
    searchFormFieldsValues: IHotelsSearchFormValues,
): HotelSlugType | undefined => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return undefined;
    }

    return toSelectedValue.redirectParams.hotelSlug;
};

const getSearchRouteUrl = ({
    activeHotelSlug,
    searchFormFieldsValues,
    searchLinkOptions,
    isSearchHotelsPage,
    hotelSlugByRouteMatch,
}: {
    activeHotelSlug?: HotelSlugType;
    searchFormFieldsValues: IHotelsSearchFormValues;
    searchLinkOptions: ISearchLinkOptions;
    isSearchHotelsPage: boolean;
    hotelSlugByRouteMatch?: HotelSlugType;
}): string => {
    const {hasDirectionSuggest} = searchLinkOptions;

    if (isSearchHotelsPage) {
        return URLs.hotelsSearch;
    }

    const hotelSlug = getHotelSlugByFields(searchFormFieldsValues);

    if (activeHotelSlug) {
        return hotelsURLs.getHotelUrlByHotelSlug(activeHotelSlug);
    }

    if (hasDirectionSuggest && hotelSlug) {
        return hotelsURLs.getHotelUrlByHotelSlug(hotelSlug);
    }

    return hotelSlugByRouteMatch
        ? hotelsURLs.getHotelUrlByHotelSlug(hotelSlugByRouteMatch)
        : URLs.hotelsHotelPage;
};

const getHotelParamsByBrowserHistory = (): ISearchLinkHotelParams => {
    const {hotelPermalink} = getQueryByBrowserHistory();

    return {
        hotelPermalink: hotelPermalink ? String(hotelPermalink) : undefined,
    };
};

const getHotelParamsByFormFields = (
    fieldsValues: IHotelsSearchFormValues,
): ISearchLinkHotelParams => {
    const toSelectedValue = fieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return {};
    }

    return {
        hotelPermalink: toSelectedValue.redirectParams.permalink,
    };
};

const getHotelPageQueryParams = (
    {
        activeHotelSlug,
        searchFormFieldsValues,
        hotelSlugByRouteMatch,
    }: {
        activeHotelSlug?: HotelSlugType;
        searchFormFieldsValues: IHotelsSearchFormValues;
        hotelSlugByRouteMatch?: HotelSlugType;
    },
    searchLinkOptions: ISearchLinkOptions,
) => {
    const {hasDirectionSuggest} = searchLinkOptions;
    const hotelSlug = getHotelSlugByFields(searchFormFieldsValues);
    const {token} = getHotelOfferToken();
    let pageHotelPermalink;
    let canSaveOfferTokenAttribution = false;

    if (hotelSlugByRouteMatch) {
        /* 1. has hotelSlug path; without permalink; save token; */
        if (hotelSlugByRouteMatch === hotelSlug) {
            canSaveOfferTokenAttribution = true;
        }
    } else {
        const {hotelPermalink} = getHotelParamsByFormFields(
            searchFormFieldsValues,
        );
        const {hotelPermalink: hotelPermalinkByQuery} =
            getHotelParamsByBrowserHistory();

        if (hasDirectionSuggest) {
            /* 2. change to hotelSlug path; without permalink; save token; */
            if (hotelPermalink === hotelPermalinkByQuery) {
                canSaveOfferTokenAttribution = true;
            }
        } else {
            /* 3. hotelPage path; with permalink; save token; */
            pageHotelPermalink = hotelPermalinkByQuery;
            canSaveOfferTokenAttribution = true;
        }
    }

    if (activeHotelSlug) {
        /* 4. Clear token and hotelPermalink; use only hotelSlug */
        canSaveOfferTokenAttribution = false;
        pageHotelPermalink = undefined;
    }

    return {
        hotelPermalink: pageHotelPermalink,
        token: canSaveOfferTokenAttribution ? token : undefined,
        ...getViewModePageParamsByQuery(),
        ...getOfferAttributionParams(),
    };
};

const getSearchPageQueryParams = (
    searchFormFieldsValues: IHotelsSearchFormValues,
) => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return {};
    }

    return {
        geoId: toSelectedValue.redirectParams.geoId,
        selectedSortId: toSelectedValue.redirectParams.selectedSortId,
        sortOrigin: toSelectedValue.redirectParams.sortOrigin,
        ...getSearchPageFilterQueryParamsByLocation(),
        ...getSearchPageLayoutViewQueryParams(),
        ...getSearchPageTopHotelParamByLocation(),
    };
};

const getBaseSearchQueryParams = (
    searchFormFieldsValues: IHotelsSearchFormValues,
): IBaseSearchQueryParams => {
    const {adults, childrenAges, startDate, endDate} = searchFormFieldsValues;

    return {
        adults,
        childrenAges: childrenAges && childrenAges.join(','),
        checkinDate: startDate ? getISObyDate(new Date(startDate)) : undefined,
        checkoutDate: endDate ? getISObyDate(new Date(endDate)) : undefined,
    };
};

const checkRegionAndSearchGroupType = (
    searchFormFieldsValues: IHotelsSearchFormValues,
    activeHotelSlug?: HotelSlugType,
): boolean => {
    const toSelectedValue = searchFormFieldsValues.to.selectedValue;

    if (typeof toSelectedValue === 'boolean') {
        return false;
    }

    const suggestGroupType = toSelectedValue.redirectParams.type;

    if (activeHotelSlug) {
        return false;
    }

    return (
        suggestGroupType === ESearchSuggestItemType.REGION ||
        suggestGroupType === ESearchSuggestItemType.HOTELS_NEARBY ||
        suggestGroupType === ESearchSuggestItemType.CROSS_SALE ||
        suggestGroupType === ESearchSuggestItemType.CROSS_SEARCH ||
        suggestGroupType === ESearchSuggestItemType.HISTORY
    );
};

export const getSearchUrl = (
    {
        activeHotelSlug,
        searchFormFieldsValues,
        hotelSlugByRouteMatch,
        queryParams,
    }: {
        activeHotelSlug?: HotelSlugType;
        searchFormFieldsValues: IHotelsSearchFormValues;
        hotelSlugByRouteMatch?: HotelSlugType;
        queryParams?: Record<string, string>;
    },
    searchLinkOptions: ISearchLinkOptions,
): string => {
    const {hasProductionOffers} = searchLinkOptions;
    const isSearchHotelsPage = checkRegionAndSearchGroupType(
        searchFormFieldsValues,
        activeHotelSlug,
    );
    const searchRouteUrl = getSearchRouteUrl({
        activeHotelSlug,
        searchFormFieldsValues,
        hotelSlugByRouteMatch,
        searchLinkOptions,
        isSearchHotelsPage,
    });

    const baseSearchParams = getBaseSearchQueryParams(searchFormFieldsValues);
    const srcParams = getStoredSrcQueryParams();
    const attributionQueryParams = getAttributionQueryParams();
    const lastSearchTimeMarkerParams = createLastSearchTimeMarker();
    const searchParamsByType = isSearchHotelsPage
        ? getSearchPageQueryParams(searchFormFieldsValues)
        : getHotelPageQueryParams(
              {
                  activeHotelSlug,
                  searchFormFieldsValues,
                  hotelSlugByRouteMatch,
              },
              searchLinkOptions,
          );

    return internalUrl(
        searchRouteUrl,
        {
            ...srcParams,
            ...baseSearchParams,
            ...searchParamsByType,
            ...attributionQueryParams,
            ...lastSearchTimeMarkerParams,
            ...queryParams,
            debugUseProdOffers: hasProductionOffers ? true : undefined,
        },
        {filterNull: true},
    );
};
