import {FunctionComponent, RefObject, useCallback, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {
    COMMON_HELP_LINK,
    TLD_PLACEHOLDER,
} from 'components/Footer/constants/footerLinks';

import {EUserAvatarSize} from 'components/User/types';
import {EHotelsExtranetGoal} from 'utilities/metrika/types/goals/hotelsExtranet';

import * as mailActions from 'reducers/common/userUnreadMailCount/actions';

import {getUserUndeadMailCount} from 'selectors/common/getUserUndeadMailCount';
import {hotelsListSelector} from 'selectors/hotelsExtranet/hotelsListSelector';

import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {useTld} from 'utilities/hooks/useTld';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useReachGoal} from 'utilities/metrika/useReachGoal';

import * as i18nBlock from 'i18n/hotelsExtranet';

import HeaderLinks from './components/HeaderLinks/HeaderLinks';
import User from 'components/User/User';
import YandexTravelLogo from 'components/YandexTravelLogo/YandexTravelLogo';
import Flex from 'components/Flex/Flex';
import Footer from './components/Footer/Footer';
import HeaderLink from './components/HeaderLink/HeaderLink';
import HotelsSelect from './components/HotelsSelect/HotelsSelect';
import KnowledgeBaseIcon from 'icons/20/KnowledgeBase';
import YandexMessengerButton from 'components/YandexMessenger/components/YandexMessengerButton/YandexMessengerButton';

import cx from './HotelsExtranetLayout.scss';

interface IHotelsExtranetLayoutProps {
    contentClassName?: string;
    authRef?: RefObject<HTMLDivElement>;
    authReturnPath?: string;
    isAuthHighlighted?: boolean;
}

const HotelsExtranetLayout: FunctionComponent<IHotelsExtranetLayoutProps> = ({
    children,
    contentClassName,
    authRef,
    authReturnPath,
    isAuthHighlighted,
}) => {
    const dispatch = useDispatch();
    const userInfo = useUserInfo();
    const unreadMailCount = useSelector(getUserUndeadMailCount);
    const fetchUserUnreadMailCount = useCallback(() => {
        dispatch(mailActions.fetchUserUnreadMailCount());
    }, [dispatch]);
    const tld = useTld();
    const {currentHotel, response} = useSelector(hotelsListSelector);
    const [isSupportVisible, setSupportVisible] = useState(false);
    const handleSupportVisibilityChange = useCallback(
        (isVisible: boolean) => {
            setSupportVisible(isVisible);
        },
        [setSupportVisible],
    );
    const {hotels = []} = response || {};
    const handleKnowledgeBaseClick = useReachGoal(
        EHotelsExtranetGoal.KNOWLEDGE_BASE_CLICK,
    );

    return (
        <Flex className={cx('root')} flexDirection="column">
            <header className={cx('header')}>
                <Flex flexDirection="row" alignItems="center">
                    <YandexTravelLogo
                        project={EProjectName.HOTELS_EXTRANET}
                        withYandexLogo
                        withoutTravelWord
                        withoutLinks={!isAuthUser(userInfo) || !hotels.length}
                    />
                    <HeaderLinks className={cx('links')} />
                </Flex>
                <Flex
                    flexDirection="row"
                    alignItems="center"
                    between={5}
                    inline
                >
                    {currentHotel && (
                        <HeaderLink
                            url={COMMON_HELP_LINK.link.replace(
                                TLD_PLACEHOLDER,
                                tld,
                            )}
                            theme="ghost"
                            onClick={handleKnowledgeBaseClick}
                        >
                            <KnowledgeBaseIcon />
                            {i18nBlock.knowledgeBase()}
                        </HeaderLink>
                    )}
                    <YandexMessengerButton
                        entrypoint="extranetHeader"
                        metrikaGoal={EHotelsExtranetGoal.SUPPORT_CHAT_CLICK}
                        additionalData={{
                            currentHotel,
                        }}
                        className={cx(
                            'supportButton',
                            isSupportVisible && 'supportButton_visible',
                        )}
                        buttonProps={{theme: 'plain'}}
                        size="l"
                        onVisibilityChange={handleSupportVisibilityChange}
                    />
                    <HotelsSelect />
                    <div ref={authRef}>
                        <User
                            userInfo={userInfo}
                            avatarSize={EUserAvatarSize.M}
                            unreadMailCount={unreadMailCount}
                            fetchUserUnreadMailCount={fetchUserUnreadMailCount}
                            authButtonTheme={
                                isAuthHighlighted ? 'primary' : undefined
                            }
                            withIcon
                            returnPath={authReturnPath}
                        />
                    </div>
                </Flex>
            </header>
            <div className={cx('content', contentClassName)}>{children}</div>
            <Footer className={cx('footer')} />
        </Flex>
    );
};

export default HotelsExtranetLayout;
