import {FunctionComponent, useCallback} from 'react';
import {generatePath, matchPath, useLocation} from 'react-router-dom';
import {useSelector} from 'react-redux';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {IExtranetHotel} from 'server/api/HotelsExtranetApi/types/IHotelsListResponse';
import {EHotelsExtranetGoal} from 'utilities/metrika/types/goals/hotelsExtranet';

import {hotelsListSelector} from 'selectors/hotelsExtranet/hotelsListSelector';

import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Flex from 'components/Flex/Flex';
import HomeIcon from 'icons/20/Home';
import ReservationsIcon from 'icons/20/Reservations';
import PaymentsIcon from 'icons/20/Payments';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';

import HeaderLink from '../HeaderLink/HeaderLink';

import cx from './HeaderLinks.scss';

function getPath(url: string, {hotelCode, partnerId}: IExtranetHotel): string {
    return generatePath(url, {hotelCode, partnerId});
}

const LINKS = [
    {
        getPath: (hotel: IExtranetHotel): string =>
            getPath(URLs.hotelsExtranetIndexPage, hotel),
        icon: <HomeIcon />,
        label: i18nBlock.main(),
        metrikaPageParam: 'main',
    },
    {
        getPath: (hotel: IExtranetHotel): string =>
            getPath(URLs.hotelsExtranetOrdersPage, hotel),
        icon: <ReservationsIcon />,
        label: i18nBlock.orders(),
        metrikaPageParam: 'orders',
    },
    {
        getPath: (hotel: IExtranetHotel): string =>
            getPath(URLs.hotelsExtranetPaymentsPage, hotel),
        icon: <PaymentsIcon />,
        label: i18nBlock.payments(),
        metrikaPageParam: 'payments',
    },
];

const HeaderLinks: FunctionComponent<IWithClassName> = ({className}) => {
    const location = useLocation();
    const {status, currentHotel} = useSelector(hotelsListSelector);
    const userInfo = useUserInfo();
    const handleLinkClick = useCallback((page): void => {
        reachGoal(EHotelsExtranetGoal.TAB_CLICK, [
            {
                hotelsExtranet: {page},
            },
        ]);
    }, []);

    if (!isAuthUser(userInfo)) {
        return null;
    }

    if (!status || status === EAsyncStatus.LOADING) {
        return (
            <Flex between={8} inline flexDirection="row" className={className}>
                <Skeleton className={cx('skeleton')} />
                <Skeleton className={cx('skeleton')} />
                <Skeleton className={cx('skeleton')} />
            </Flex>
        );
    }

    if (!currentHotel) {
        return null;
    }

    return (
        <Flex between={8} inline flexDirection="row" className={className}>
            {LINKS.map(({getPath, icon, label, metrikaPageParam}) => {
                const to = getPath(currentHotel);

                return (
                    <HeaderLink
                        key={to}
                        to={to}
                        theme={
                            matchPath(location.pathname, {
                                path: to,
                                exact: true,
                            })
                                ? 'black'
                                : 'ghost'
                        }
                        onClick={(): void => handleLinkClick(metrikaPageParam)}
                    >
                        {icon}
                        {label}
                    </HeaderLink>
                );
            })}
        </Flex>
    );
};

export default HeaderLinks;
