import {FC, useCallback, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {batchActions} from 'redux-batched-actions';
import {generatePath, useHistory, useRouteMatch} from 'react-router-dom';

import {URLs} from 'constants/urls';

import EAsyncStatus from 'types/common/EAsyncStatus';
import {IExtranetHotel} from 'server/api/HotelsExtranetApi/types/IHotelsListResponse';

import {resetUpcomingArrivals} from 'reducers/hotelsExtranet/upcomingArrivals/actions';
import {resetOrders} from 'reducers/hotelsExtranet/orders/actions';
import {resetPayments} from 'reducers/hotelsExtranet/payments/actions';
import {setSelectedHotel} from 'reducers/hotelsExtranet/hotelsList/thunk';

import {hotelsListSelector} from 'selectors/hotelsExtranet/hotelsListSelector';

import {getHotelId} from 'projects/hotelsExtranet/utilities/getHotelId';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Select, {ISelectOption} from 'components/Select/Select';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';

import cx from './HotelsSelect.scss';

function mapHotel(hotel: IExtranetHotel): ISelectOption<string, string> {
    return {
        value: getHotelId(hotel),
        data: hotel.name || getHotelId(hotel),
    };
}

const HotelsSelect: FC = () => {
    const dispatch = useDispatch();
    const {status, response, currentHotel} = useSelector(hotelsListSelector);
    const {path} = useRouteMatch();
    const history = useHistory();
    const {hotels} = response || {};
    const userInfo = useUserInfo();

    const handleHotelChange = useCallback(
        value => {
            const newHotel = hotels?.find(h => getHotelId(h) === value);

            if (newHotel) {
                dispatch(setSelectedHotel(newHotel));
                dispatch(
                    batchActions([
                        resetUpcomingArrivals(),
                        resetOrders(),
                        resetPayments(),
                    ]),
                );

                if (newHotel) {
                    const {hotelCode, partnerId} = newHotel;
                    const newPath = generatePath(
                        path === URLs.hotelsExtranetRoot
                            ? URLs.hotelsExtranetIndexPage
                            : path,
                        {hotelCode, partnerId},
                    );

                    history.replace(newPath);
                }
            }
        },
        [dispatch, hotels, history, path],
    );

    const options = useMemo(() => {
        const result = (hotels || []).map(mapHotel);

        if (!currentHotel) {
            result.push({
                data: i18nBlock.hotelNotSelected(),
                value: '',
            });
        } else if (
            !hotels?.some(
                h =>
                    h.hotelCode === currentHotel.hotelCode &&
                    h.partnerId === currentHotel.partnerId,
            )
        ) {
            result.push(mapHotel(currentHotel));
        }

        return result;
    }, [hotels, currentHotel]);

    if (!isAuthUser(userInfo)) {
        return null;
    }

    if (!status || status === EAsyncStatus.LOADING) {
        return <Skeleton className={cx('hotelsSkeleton')} />;
    }

    if (!hotels?.length) {
        return null;
    }

    return (
        <Select
            options={options}
            value={currentHotel ? getHotelId(currentHotel) : ''}
            onChange={handleHotelChange}
            theme="plain"
            className={cx('hotelsList')}
            menuClassName={cx('hotelsListMenu')}
        />
    );
};

export default HotelsSelect;
