import {FunctionComponent} from 'react';

import {IWithClassName} from 'types/withClassName';

import Flex from 'components/Flex/Flex';
import Select from 'components/Select/Select';
import Text from 'components/Text/Text';
import PaginationInput from 'components/PaginationInput/PaginationInput';

import cx from './Paginator.scss';

const PAGE_SIZES_COUNT = [
    {value: 10, data: '10'},
    {value: 25, data: '25'},
    {value: 50, data: '50'},
];

export const DEFAULT_PAGE_SIZE = PAGE_SIZES_COUNT[0].value;

interface IPaginatorProps extends IWithClassName {
    pageNumber: number;
    pageSize?: number;
    pageSizeLabel: string;
    pageCount: number;
    onPageSizeChange(pageSize: number): void;
    onPageSizeChangeClick(): void;
    onPageChange(page: number): void;
    onPageInput(): void;
    onPageArrowClick(): void;
}

const Paginator: FunctionComponent<IPaginatorProps> = ({
    className,
    pageNumber,
    pageSize,
    pageSizeLabel,
    pageCount,
    onPageSizeChange,
    onPageSizeChangeClick,
    onPageChange,
    onPageInput,
    onPageArrowClick,
}) => (
    <Flex
        className={className}
        flexDirection="row"
        alignItems="center"
        justifyContent="space-between"
    >
        <Flex flexDirection="row" alignItems="center" between={2} inline>
            <Select
                menuClassName={cx('pageSize')}
                theme="outlined"
                onChange={onPageSizeChange}
                onShowPopup={onPageSizeChangeClick}
                options={PAGE_SIZES_COUNT}
                value={pageSize || DEFAULT_PAGE_SIZE}
                size="m"
            />
            <Text>{pageSizeLabel}</Text>
        </Flex>
        <PaginationInput
            minValue={1}
            maxValue={pageCount}
            pageNumber={pageNumber}
            onPageChange={onPageChange}
            onPageInput={onPageInput}
            onArrowClick={onPageArrowClick}
        />
    </Flex>
);

export default Paginator;
