import {FC} from 'react';
import moment from 'moment';

import {IWithClassName} from 'types/withClassName';
import IPrice from 'types/common/price/IPrice';

import {ROBOT} from 'utilities/dateUtils/formats';
import {CHAR_EM_DASH} from 'utilities/strings/charCodes';
import {formatPrice} from 'projects/hotelsExtranet/utilities/formatPrice';

import Flex from 'components/Flex/Flex';

import cx from './Table.scss';

const Table: FC<IWithClassName> = ({className, children}) => (
    <table className={cx('table', className)} cellSpacing="0">
        {children}
    </table>
);

export const TableHead: FC = ({children}) => {
    return <thead>{children}</thead>;
};

export const TableBody: FC = ({children}) => {
    return <tbody>{children}</tbody>;
};

export const TableFooter: FC = ({children}) => {
    return <tfoot className={cx('footer')}>{children}</tfoot>;
};

export const TableRow: FC = ({children}) => {
    return <tr>{children}</tr>;
};

type TTableHeadCellProps = IWithClassName & {
    onClick?: () => void;
    noSideBorders?: boolean;
    withSort?: boolean;
};

export const TableHeadCell: FC<TTableHeadCellProps> = ({
    className,
    children,
    onClick,
    noSideBorders,
    withSort,
}) => {
    return (
        <th
            className={cx(
                'headCell',
                {
                    headCell_noSideBorders: noSideBorders,
                    headCell_withSort: withSort,
                },
                className,
            )}
            onClick={onClick}
        >
            <Flex alignItems="center" className={cx('headCellWrap')}>
                {children}
            </Flex>
        </th>
    );
};

type TTableCellProps = IWithClassName & {
    wrapClassName?: string;
    noSideBorders?: boolean;
};

export const TableCell: FC<TTableCellProps> = ({
    className,
    wrapClassName,
    children,
    noSideBorders,
}) => {
    return (
        <td
            className={cx(
                'tableCell',
                {
                    headCell_noSideBorders: noSideBorders,
                },
                className,
            )}
        >
            <Flex
                className={cx('tableCellWrap', wrapClassName)}
                alignItems="center"
            >
                {children}
            </Flex>
        </td>
    );
};

export const TableDateCell: FC<TTableCellProps & {date?: string}> = ({
    date,
    ...props
}) => {
    return (
        <TableCell {...props}>
            {date ? moment(date).format(ROBOT) : CHAR_EM_DASH}
        </TableCell>
    );
};

export const TablePriceCell: FC<TTableCellProps & {price?: IPrice}> = ({
    price,
    ...props
}) => {
    return <TableCell {...props}>{formatPrice(price)}</TableCell>;
};

export const TableFooterCell: FC<IWithClassName & {colSpan?: number}> = ({
    className,
    children,
    colSpan,
}) => {
    return (
        <td className={cx('footerCell', className)} colSpan={colSpan}>
            {children}
        </td>
    );
};

export default Table;
