import {FunctionComponent, useCallback, useMemo, useRef, useState} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import EAsyncStatus from 'types/common/EAsyncStatus';
import EPopupDirection from 'components/Popup/types/EPopupDirection';
import {IHotelsExtranetAddHotelRequest} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetAddHotelRequest';

import {addHotel} from 'reducers/hotelsExtranet/addHotel/thunk';

import {addHotelSelector} from 'selectors/hotelsExtranet/addHotelSelector';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {setFromAuth} from 'projects/hotelsExtranet/utilities/useAddFormQueryParams';

import * as i18nBlock from 'i18n/hotelsExtranet';

import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import HotelsExtranetLayout from 'projects/hotelsExtranet/components/HotelsExtranetLayout/HotelsExtranetLayout';
import Text from 'components/Text/Text';
import RoundButton from 'components/RoundButton/RoundButton';
import ArrowLeftIcon from 'icons/24/ArrowLeft';
import RequestResultPage from './components/RequestResultPage/RequestResultPage';
import useReturnPathUrl from 'components/User/hooks/useReturnPathUrl';
import AddHotelForm from './components/AddHotelForm/AddHotelForm';

import cx from './AddHotelPage.scss';

const AddHotelPage: FunctionComponent = () => {
    const history = useHistory();
    const dispatch = useDispatch();
    const {status} = useSelector(addHotelSelector);
    const authRef = useRef<HTMLDivElement>(null);
    const [authParams, setAuthParams] =
        useState<Partial<IHotelsExtranetAddHotelRequest>>();
    const {
        value: isForceLoginVisible,
        setFalse: hideForceLogin,
        setTrue: showForceLogin,
    } = useBoolean(true);
    const {
        value: isAuthHighlighted,
        setTrue: highlightAuth,
        setFalse: unhighlightAuth,
    } = useBoolean(false);
    const user = useUserInfo();
    const toolTipMessage = useMemo((): string => {
        if (isAuthUser(user)) {
            return i18nBlock.addToAccountTooltip({
                login: String(user.login),
            });
        }

        return i18nBlock.forceLoginTooltip();
    }, [user]);

    const authReturnPath = useReturnPathUrl(authParams);

    const handleFormChange = useCallback(
        values => {
            const prepared: Record<string, unknown> = {};

            Object.entries(values).forEach(([key, value]) => {
                if (value) {
                    prepared[key] = value;
                }
            });

            setFromAuth(prepared);

            setAuthParams(prepared);
        },
        [setAuthParams],
    );
    const handleBackClick = useCallback(() => history.goBack(), [history]);
    const handleSubmit = useCallback(
        values => {
            if (isAuthUser(user)) {
                hideForceLogin();
                unhighlightAuth();
                dispatch(addHotel(values));
            } else {
                highlightAuth();
                showForceLogin();
            }
        },
        [
            dispatch,
            user,
            highlightAuth,
            unhighlightAuth,
            showForceLogin,
            hideForceLogin,
        ],
    );

    if (status === EAsyncStatus.SUCCESS) {
        return <RequestResultPage />;
    }

    return (
        <HotelsExtranetLayout
            contentClassName={cx('layout')}
            authRef={authRef}
            authReturnPath={authReturnPath}
            isAuthHighlighted={isAuthHighlighted}
        >
            <div className={cx('panel')}>
                <RoundButton
                    className={cx('backButton')}
                    onClick={handleBackClick}
                    size="l"
                    icon={<ArrowLeftIcon />}
                ></RoundButton>
                <AddHotelForm
                    onChange={handleFormChange}
                    onSubmit={handleSubmit}
                />
            </div>
            <MessageBoxPopup
                className={cx('messageBox')}
                isVisible={isForceLoginVisible}
                anchorRef={authRef}
                direction={EPopupDirection.BOTTOM}
                hasClose
                onClose={hideForceLogin}
            >
                <Text className={cx('messageBoxText')} tag="div" size="m">
                    {toolTipMessage}
                </Text>
            </MessageBoxPopup>
        </HotelsExtranetLayout>
    );
};

export default AddHotelPage;
