import {FunctionComponent, ReactNode, useCallback} from 'react';
import {FormSpy} from 'react-final-form';

import EHotelPartnerId from 'server/api/GenericOrderApi/types/common/service/IHotelServiceInfo/IHotelOfferInfo/EHotelPartnerId';
import {
    EHotelsExtranetAddHotelRequestFieldName,
    IHotelsExtranetAddHotelRequest,
} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetAddHotelRequest';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useAddFormQueryParams} from 'projects/hotelsExtranet/utilities/useAddFormQueryParams';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Text from 'components/Text/Text';
import RadioButton from 'components/RadioButtonSlide/RadioButtonSlide';
import Input from 'components/Input/Input';
import Button from 'components/Button/Button';
import Form from 'components/Form/Form';
import FormField from 'components/FormField/FormField';
import Heading from 'components/Heading/Heading';

import {validationInfo} from './validationInfo';

import cx from './AddHotelForm.scss';

interface IAddHotelFormProps {
    onChange: (values: Partial<IHotelsExtranetAddHotelRequest>) => void;
    onSubmit: (values: IHotelsExtranetAddHotelRequest) => void;
}

function getPartnerName(partnerId?: EHotelPartnerId): string {
    if (!partnerId) {
        return '';
    }

    switch (partnerId) {
        case EHotelPartnerId.PI_TRAVELLINE:
            return i18nBlock.travelLine();
        case EHotelPartnerId.PI_BNOVO:
            return i18nBlock.bnovo();
    }

    return partnerId as string;
}

const AddHotelForm: FunctionComponent<IAddHotelFormProps> = ({
    onChange,
    onSubmit,
}) => {
    const deviceType = useDeviceType();
    const {
        prefill: {email, hotelCode, partnerID},
    } = useAddFormQueryParams();

    const initialPartnerId =
        partnerID === EHotelPartnerId.PI_BNOVO
            ? EHotelPartnerId.PI_BNOVO
            : EHotelPartnerId.PI_TRAVELLINE;

    const handleFormChange = useCallback(
        form => onChange(form.values),
        [onChange],
    );

    return (
        <Form
            onSubmit={onSubmit}
            validationInfo={validationInfo}
            initialValues={{
                [EHotelsExtranetAddHotelRequestFieldName.PARTNER_ID]:
                    initialPartnerId,
                [EHotelsExtranetAddHotelRequestFieldName.EMAIL]: email,
                [EHotelsExtranetAddHotelRequestFieldName.HOTEL_CODE]: hotelCode,
            }}
            render={({handleSubmit, form}): ReactNode => {
                const partnerId = form.getFieldState(
                    EHotelsExtranetAddHotelRequestFieldName.PARTNER_ID,
                )?.value;
                const partner = getPartnerName(partnerId);

                return (
                    <form
                        className={cx('form')}
                        onSubmit={handleSubmit}
                        autoComplete="off"
                    >
                        <Heading level="2">
                            {i18nBlock.enterYourHotelDetails()}
                        </Heading>
                        <Text className={cx('label')}>
                            {i18nBlock.bookingSystem()}
                        </Text>
                        <FormField
                            name={
                                EHotelsExtranetAddHotelRequestFieldName.PARTNER_ID
                            }
                            deviceType={deviceType}
                            control={({input}): ReactNode => (
                                <RadioButton
                                    onChange={input.onChange}
                                    backgroundClassName={cx('radio')}
                                    value={input.value}
                                >
                                    <RadioButton.Option
                                        value={EHotelPartnerId.PI_TRAVELLINE}
                                    >
                                        {getPartnerName(
                                            EHotelPartnerId.PI_TRAVELLINE,
                                        )}
                                    </RadioButton.Option>
                                    <RadioButton.Option
                                        value={EHotelPartnerId.PI_BNOVO}
                                    >
                                        {getPartnerName(
                                            EHotelPartnerId.PI_BNOVO,
                                        )}
                                    </RadioButton.Option>
                                </RadioButton>
                            )}
                        />
                        <Text className={cx('label')}>
                            {i18nBlock.idInTheBookingSystem({partner})}
                        </Text>
                        <FormField
                            name={
                                EHotelsExtranetAddHotelRequestFieldName.HOTEL_CODE
                            }
                            deviceType={deviceType}
                            control={(
                                {input},
                                {error, inputRef, controlRef},
                            ): ReactNode => (
                                <Input
                                    {...input}
                                    size="l"
                                    innerRef={controlRef}
                                    inputRef={(currentInputRef): void => {
                                        inputRef.current = currentInputRef;
                                    }}
                                    state={error ? 'error' : undefined}
                                />
                            )}
                        />
                        <Text className={cx('label')}>
                            {i18nBlock.hotelEmailInTheBookingSystem({partner})}
                        </Text>
                        <FormField
                            name={EHotelsExtranetAddHotelRequestFieldName.EMAIL}
                            deviceType={deviceType}
                            control={(
                                {input},
                                {error, inputRef, controlRef},
                            ): ReactNode => (
                                <Input
                                    {...input}
                                    size="l"
                                    innerRef={controlRef}
                                    inputRef={(currentInputRef): void => {
                                        inputRef.current = currentInputRef;
                                    }}
                                    state={error ? 'error' : undefined}
                                />
                            )}
                        />
                        <Button
                            className={cx('submit')}
                            type="submit"
                            size="l"
                            theme="primary"
                        >
                            {i18nBlock.requestAccess()}
                        </Button>
                        <FormSpy onChange={handleFormChange} />
                    </form>
                );
            }}
        />
    );
};

export default AddHotelForm;
