import {FC, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import moment from 'moment';

import EAsyncStatus from 'types/common/EAsyncStatus';

import {fetchUpcomingArrivals} from 'reducers/hotelsExtranet/upcomingArrivals/thunk';

import {upcomingArrivalsSelector} from 'selectors/hotelsExtranet/upcomingArrivalsSelector';
import {hotelsListSelector} from 'selectors/hotelsExtranet/hotelsListSelector';

import {ROBOT} from 'utilities/dateUtils/formats';

import * as i18nBlock from 'i18n/hotelsExtranet';

import OrdersTable from '../../../OrdersPage/components/OrdersTable/OrdersTable';
import Spinner from 'components/Spinner/Spinner';
import Text from 'components/Text/Text';

import cx from './UpcomingArrivals.scss';

const ORDERS_LIMIT = 5;

const UpcomingArrivals: FC = () => {
    const dispatch = useDispatch();
    const {response, status} = useSelector(upcomingArrivalsSelector);
    const {currentHotel} = useSelector(hotelsListSelector);

    const {result} = response || {};

    useEffect(() => {
        if (!status && currentHotel) {
            dispatch(
                fetchUpcomingArrivals({
                    originalId: currentHotel.hotelCode,
                    partnerId: currentHotel.partnerId,
                    sortFields: [
                        {
                            field: 'checkInDate',
                            sortDirection: 'SD_ASC',
                        },
                    ],
                    checkInDate: {
                        from: moment().format(ROBOT),
                        till: moment().add(1, 'year').format(ROBOT),
                    },
                    page: {
                        num: 0,
                        limit: ORDERS_LIMIT,
                    },
                }),
            );
        }
    }, [dispatch, status, currentHotel]);

    return (
        <div className={cx('root')}>
            {!status ||
                (status === EAsyncStatus.LOADING && (
                    <Spinner className={cx('spinner')} />
                ))}
            {status === EAsyncStatus.ERROR && (
                <Text color="alert">{i18nBlock.failedToLoadOrders()}</Text>
            )}
            {status === EAsyncStatus.SUCCESS && Number(result?.length) > 0 && (
                <OrdersTable
                    className={cx('table')}
                    headCellClassName={cx('headCell')}
                    items={result || []}
                    isShortView
                />
            )}
            {status === EAsyncStatus.SUCCESS &&
                Number(result?.length) === 0 && (
                    <Text>{i18nBlock.noData()}</Text>
                )}
        </div>
    );
};

export default UpcomingArrivals;
