import {FunctionComponent, useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import moment from 'moment';

import {IWithClassName} from 'types/withClassName';
import {IDateFilter} from 'server/api/HotelsExtranetApi/types/IDateFilter';
import {
    IHotelsExtranetOrdersFilters,
    IHotelsExtranetOrdersRequest,
} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetOrdersRequest';
import {EHotelsExtranetGoal} from 'utilities/metrika/types/goals/hotelsExtranet';

import {setFilters} from 'reducers/hotelsExtranet/orders/thunk';

import {ordersSelector} from 'selectors/hotelsExtranet/ordersSelector';

import {ROBOT} from 'utilities/dateUtils/formats';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {params} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Flex from 'components/Flex/Flex';
import Select from 'components/Select/Select';
import Button from 'components/Button/Button';
import ArrowBottomIcon from 'icons/16/ArrowBottom';
import DatePicker from 'projects/hotelsExtranet/components/DatePicker/DatePicker';

import cx from './Filters.scss';

const DATE_TYPES = [
    {value: 'checkInDate', data: i18nBlock.checkIn()},
    {value: 'checkOutDate', data: i18nBlock.checkOut()},
    {value: 'cancelledAt', data: i18nBlock.cancelledAt()},
    {value: 'orderedAt', data: i18nBlock.bookingDate()},
];
const DEFAULT_DATE_TYPE = DATE_TYPES[0].value;

function getSelectedFilter(request?: IHotelsExtranetOrdersRequest): {
    name?: string;
    value?: IDateFilter;
} {
    if (!request) {
        return {};
    }

    const entries = Object.entries(request);

    const selectedDate = entries.find(([key, value]) =>
        DATE_TYPES.some(dt => dt.value === key && value),
    );

    if (!selectedDate) {
        return {};
    }

    return {
        name: selectedDate[0],
        value: selectedDate[1] as IDateFilter,
    };
}

function clearFilters(): IHotelsExtranetOrdersFilters {
    return DATE_TYPES.reduce(
        (result, {value}) => ({...result, [value]: null}),
        {},
    );
}

const Filters: FunctionComponent<IWithClassName> = ({className}) => {
    const dispatch = useDispatch();
    const [areDatesRequired, setDatesRequired] = useState(false);

    const {request} = useSelector(ordersSelector);

    const requestFilter = getSelectedFilter(request);

    const [dateType, setDateType] = useState(
        requestFilter.name ? requestFilter.name : DEFAULT_DATE_TYPE,
    );
    const [startDate, setStartDate] = useState(
        requestFilter.value ? moment(requestFilter.value.from).toDate() : null,
    );
    const [endDate, setEndDate] = useState(
        requestFilter.value ? moment(requestFilter.value.till).toDate() : null,
    );

    const handleShowCalendar = useReachGoal(
        EHotelsExtranetGoal.CALENDAR_CLICK,
        [
            {
                hotelsExtranet: {
                    page: 'orders',
                },
            },
        ],
    );

    const handleDateTypeChange = useCallback(
        value => {
            setDateType(value);
        },
        [setDateType],
    );

    const handleStartDateSelect = useCallback(
        ({startDate, endDate}) => {
            setStartDate(startDate);
            setEndDate(endDate);
        },
        [setStartDate, setEndDate],
    );

    const handleSubmitClick = useCallback(() => {
        if (!startDate) {
            setDatesRequired(true);

            return;
        }

        const filters = {
            ...clearFilters(),
            [dateType]: {
                from: moment(startDate).format(ROBOT),
                till: moment(endDate).format(ROBOT),
            },
        };

        dispatch(setFilters(filters));
        params({
            hotelsExtranet: {
                filters,
            },
        });
    }, [dispatch, dateType, startDate, endDate, setDatesRequired]);

    const handleResetClick = useCallback(() => {
        setDatesRequired(false);
        setStartDate(null);
        setEndDate(null);
        setDateType(DEFAULT_DATE_TYPE);
        dispatch(
            setFilters({
                ...clearFilters(),
            }),
        );
    }, [setDatesRequired, setStartDate, setEndDate, setDateType, dispatch]);

    useEffect(() => setDatesRequired(false), [startDate, endDate]);

    return (
        <Flex
            className={className}
            between={2}
            inline
            alignItems="flex-end"
            flexDirection="row"
        >
            <Select
                className={cx('dateType')}
                onChange={handleDateTypeChange}
                options={DATE_TYPES}
                value={dateType}
                size="m"
                icon={ArrowBottomIcon}
            />
            <DatePicker
                startDate={startDate}
                endDate={endDate}
                onShowCalendar={handleShowCalendar}
                onDaySelect={handleStartDateSelect}
                areDatesRequired={areDatesRequired}
            />
            <Button
                theme="primary"
                size="m"
                className={cx('submit')}
                onClick={handleSubmitClick}
            >
                {i18nBlock.show()}
            </Button>
            <Button
                theme="secondary"
                size="m"
                className={cx('submit')}
                onClick={handleResetClick}
                disabled={!startDate}
            >
                {i18nBlock.clear()}
            </Button>
        </Flex>
    );
};

export default Filters;
