import {Fragment, FunctionComponent, ReactNode, useCallback} from 'react';
import {useDispatch} from 'react-redux';

import IPrice from 'types/common/price/IPrice';
import {IWithClassName} from 'types/withClassName';
import {IExtranetOrderInfo} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetOrdersResponse';
import {ISortBy} from 'server/api/HotelsExtranetApi/types/ISortBy';
import {TSortFieldName} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetOrdersRequest';

import {setSortField} from 'reducers/hotelsExtranet/orders/thunk';

import {getOrderStatus} from 'projects/hotelsExtranet/utilities/getOrderStatus';
import {getNewSortBy} from 'projects/hotelsExtranet/utilities/getNewSortBy';
import {formatPrice} from 'projects/hotelsExtranet/utilities/formatPrice';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';
import Text from 'components/Text/Text';
import Table, {
    TableBody,
    TableCell,
    TableDateCell,
    TableFooter,
    TableFooterCell,
    TableHead,
    TableHeadCell,
    TablePriceCell,
    TableRow,
} from 'projects/hotelsExtranet/components/Table/Table';
import SortIcon from 'projects/hotelsExtranet/components/SortIcon/SortIcon';

import cx from './OrdersTable.scss';

interface IColumn {
    title: string;
    name?: TSortFieldName;
    isInShortView?: true;
    renderCell: (item: IExtranetOrderInfo) => ReactNode;
}

const COLUMNS: IColumn[] = [
    {
        title: i18nBlock.guestName(),
        isInShortView: true,
        renderCell: item => (
            <TableCell>
                {item.firstGuest.firstName} {item.firstGuest.lastName}
            </TableCell>
        ),
    },
    {
        title: i18nBlock.checkIn(),
        name: 'checkInDate',
        isInShortView: true,
        renderCell: item => <TableDateCell date={item.checkInDate} />,
    },
    {
        title: i18nBlock.checkOut(),
        name: 'checkOutDate',
        isInShortView: true,
        renderCell: item => <TableDateCell date={item.checkOutDate} />,
    },
    {
        title: i18nBlock.bookingDate(),
        name: 'createdAt',
        renderCell: item => <TableDateCell date={item.createdAt} />,
    },
    {
        title: i18nBlock.bookingStatus(),
        renderCell: item => (
            <TableCell>{getOrderStatus(item.statusLocalized)}</TableCell>
        ),
    },
    {
        title: i18nBlock.paidByGuest(),
        name: 'fiscalPrice',
        isInShortView: true,
        renderCell: item => (
            <TablePriceCell price={item.priceBreakdown.fiscalPrice} />
        ),
    },
    {
        title: i18nBlock.paidToHotel(),
        name: 'partnerAmount',
        renderCell: item => (
            <TablePriceCell price={item.priceBreakdown.partner} />
        ),
    },
    {
        title: i18nBlock.yandexAndPartnerId(),
        isInShortView: true,
        renderCell: item => (
            <TableCell wrapClassName={cx('idsCell')}>
                {item.id && (
                    <Text whiteSpace="nowrap" tag="div">
                        {item.prettyId}
                        {item.partnerOrderId && ` /`}
                    </Text>
                )}
                <Text whiteSpace="nowrap" tag="div">
                    {item.partnerOrderId}
                </Text>
            </TableCell>
        ),
    },
    {
        title: i18nBlock.paymentNumber(),
        renderCell: item => (
            <TableCell>
                {item.bankOrderInfo?.map(b => b.bankOrderId).join(', ')}
            </TableCell>
        ),
    },
];

interface IOrdersTableProps extends IWithClassName {
    items: IExtranetOrderInfo[];
    sortBy?: ISortBy<TSortFieldName>;
    fiscalSum?: IPrice;
    partnerSum?: IPrice;
    isShortView?: boolean;
    withSort?: boolean;
    withSpinner?: boolean;
    headCellClassName?: string;
}

const OrdersTable: FunctionComponent<IOrdersTableProps> = ({
    className,
    headCellClassName,
    items,
    sortBy,
    fiscalSum,
    partnerSum,
    isShortView,
    withSort,
    withSpinner,
}) => {
    const dispatch = useDispatch();

    const handleHeadClick = useCallback(
        (fieldName?: TSortFieldName): void => {
            if (!fieldName) {
                return;
            }

            const newSortBy = getNewSortBy<TSortFieldName>(fieldName, sortBy);

            dispatch(setSortField(newSortBy));
        },
        [dispatch, sortBy],
    );

    function renderFooter(): ReactNode {
        if (withSpinner) {
            return (
                <TableFooter>
                    <TableRow>
                        <TableFooterCell colSpan={9}>
                            <Flex alignItems="center" justifyContent="center">
                                <Spinner size="xxs" />
                                <span className={cx('loadingText')}>
                                    {i18nBlock.loadingOrders()}
                                </span>
                            </Flex>
                        </TableFooterCell>
                    </TableRow>
                </TableFooter>
            );
        }

        if (!isShortView && (partnerSum || fiscalSum)) {
            return (
                <TableFooter>
                    <TableRow>
                        <TableFooterCell colSpan={5}>
                            {i18nBlock.total()}
                        </TableFooterCell>
                        <TableFooterCell>
                            {fiscalSum ? formatPrice(fiscalSum) : null}
                        </TableFooterCell>
                        <TableFooterCell>
                            {partnerSum ? formatPrice(partnerSum) : null}
                        </TableFooterCell>
                        <TableFooterCell colSpan={2}></TableFooterCell>
                    </TableRow>
                </TableFooter>
            );
        }

        return null;
    }

    return (
        <Table className={className}>
            <TableHead>
                <TableRow>
                    {COLUMNS.map(
                        ({title, name, isInShortView}) =>
                            (!isShortView || isInShortView) && (
                                <TableHeadCell
                                    className={headCellClassName}
                                    key={title}
                                    onClick={(): void => handleHeadClick(name)}
                                    withSort={name && withSort}
                                >
                                    {title}
                                    {withSort && name && (
                                        <SortIcon
                                            columnName={name}
                                            sortField={sortBy?.field}
                                            sortDirection={
                                                sortBy?.sortDirection
                                            }
                                        />
                                    )}
                                </TableHeadCell>
                            ),
                    )}
                </TableRow>
            </TableHead>
            <TableBody>
                {items.map(item => (
                    <TableRow key={item.id}>
                        {COLUMNS.map(({title, renderCell, isInShortView}) => (
                            <Fragment key={title}>
                                {(!isShortView || isInShortView) &&
                                    renderCell(item)}
                            </Fragment>
                        ))}
                    </TableRow>
                ))}
            </TableBody>
            {renderFooter()}
        </Table>
    );
};

export default OrdersTable;
