import {FunctionComponent, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import EAsyncStatus from 'types/common/EAsyncStatus';

import {fetchPayments} from 'reducers/hotelsExtranet/payments/thunk';

import {paymentsSelector} from 'selectors/hotelsExtranet/paymentsSelector';
import {hotelsListSelector} from 'selectors/hotelsExtranet/hotelsListSelector';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Heading from 'components/Heading/Heading';
import HotelsExtranetLayout from 'projects/hotelsExtranet/components/HotelsExtranetLayout/HotelsExtranetLayout';
import PaymentsTable from './components/PaymentsTable/PaymentsTable';
import Spinner from 'components/Spinner/Spinner';
import PaymentsPaginator from './components/PaymentsPaginator/PaymentsPaginator';
import Filters from './components/Filters/Filters';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import {DEFAULT_PAGE_SIZE} from 'projects/hotelsExtranet/components/Paginator/Paginator';

import cx from './PaymentsPage.scss';

const PaymentsPage: FunctionComponent = () => {
    const dispatch = useDispatch();
    const {response, request, status} = useSelector(paymentsSelector);
    const {currentHotel} = useSelector(hotelsListSelector);
    const {sortFields} = request || {};
    const {result, sum} = response || {};

    useEffect(() => {
        if (!status && currentHotel) {
            dispatch(
                fetchPayments({
                    originalId: currentHotel.hotelCode,
                    partnerId: currentHotel.partnerId,
                    page: {
                        num: 0,
                        limit: DEFAULT_PAGE_SIZE,
                    },
                }),
            );
        }
    }, [dispatch, status, currentHotel]);

    return (
        <HotelsExtranetLayout>
            <Heading className={cx('title')} level="1">
                <Flex justifyContent="space-between" alignItems="center">
                    {i18nBlock.payments()}
                </Flex>
            </Heading>
            {status === EAsyncStatus.LOADING && !result && (
                <Spinner className={cx('spinner')} />
            )}
            {status === EAsyncStatus.ERROR && (
                <Text className={cx('error')} color="alert">
                    {i18nBlock.failedToLoadPayments()}
                </Text>
            )}
            {result && (
                <>
                    <Filters className={cx('filters')} />
                    {result.length === 0 ? (
                        <Text className={cx('noData')} tag="div">
                            {i18nBlock.noData()}
                        </Text>
                    ) : (
                        <>
                            <PaymentsTable
                                className={cx('table')}
                                items={result}
                                total={sum}
                                sortBy={sortFields ? sortFields[0] : undefined}
                                withSpinner={status === EAsyncStatus.LOADING}
                            />
                            <PaymentsPaginator className={cx('paginator')} />
                        </>
                    )}
                </>
            )}
        </HotelsExtranetLayout>
    );
};

export default PaymentsPage;
