import moment from 'moment';
import {FunctionComponent, useCallback, useEffect, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {EHotelsExtranetGoal} from 'utilities/metrika/types/goals/hotelsExtranet';

import {setFilters} from 'reducers/hotelsExtranet/payments/thunk';

import {paymentsSelector} from 'selectors/hotelsExtranet/paymentsSelector';

import {ROBOT} from 'utilities/dateUtils/formats';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {params} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import DatePicker from 'projects/hotelsExtranet/components/DatePicker/DatePicker';

import cx from './Filters.scss';

const Filters: FunctionComponent<IWithClassName> = ({className}) => {
    const dispatch = useDispatch();
    const [areDatesRequired, setDatesRequired] = useState(false);

    const {request} = useSelector(paymentsSelector);

    const handleShowCalendar = useReachGoal(
        EHotelsExtranetGoal.CALENDAR_CLICK,
        [
            {
                hotelsExtranet: {
                    page: 'payments',
                },
            },
        ],
    );

    const [startDate, setStartDate] = useState(
        request?.eventTime?.from
            ? moment(request?.eventTime?.from).toDate()
            : null,
    );
    const [endDate, setEndDate] = useState(
        request?.eventTime?.till
            ? moment(request?.eventTime?.till).toDate()
            : null,
    );

    const handleStartDateSelect = useCallback(({startDate, endDate}) => {
        setStartDate(startDate);
        setEndDate(endDate);
    }, []);

    const handleSubmitClick = useCallback(() => {
        if (!startDate) {
            setDatesRequired(true);

            return;
        }

        const filters = {
            from: moment(startDate).format(ROBOT),
            till: moment(endDate || startDate).format(ROBOT),
        };

        dispatch(setFilters(filters));
        params({
            hotelsExtranet: {
                filters,
            },
        });
    }, [dispatch, startDate, endDate, setDatesRequired]);

    const handleResetClick = useCallback(() => {
        setDatesRequired(false);
        setStartDate(null);
        setEndDate(null);

        dispatch(setFilters());
    }, [setDatesRequired, setStartDate, setEndDate, dispatch]);

    useEffect(() => setDatesRequired(false), [startDate, endDate]);

    return (
        <Flex
            className={cx('root', className)}
            between={2}
            inline
            alignItems="flex-end"
        >
            <DatePicker
                startDate={startDate}
                endDate={endDate}
                onShowCalendar={handleShowCalendar}
                onDaySelect={handleStartDateSelect}
                areDatesRequired={areDatesRequired}
            />
            <Button
                theme="primary"
                size="m"
                className={cx('submit')}
                onClick={handleSubmitClick}
            >
                {i18nBlock.show()}
            </Button>
            <Button
                theme="secondary"
                size="m"
                className={cx('submit')}
                onClick={handleResetClick}
                disabled={!startDate}
            >
                {i18nBlock.clear()}
            </Button>
        </Flex>
    );
};

export default Filters;
