import {FunctionComponent, ReactNode, useCallback, useMemo} from 'react';
import {useDispatch} from 'react-redux';

import {
    IExtranetPaymentInfo,
    TSortFieldName,
} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetPaymentsResponse';
import {IWithClassName} from 'types/withClassName';
import IPrice from 'types/common/price/IPrice';
import {ISortBy} from 'server/api/HotelsExtranetApi/types/ISortBy';

import {setSortField} from 'reducers/hotelsExtranet/payments/thunk';

import {formatPrice} from 'projects/hotelsExtranet/utilities/formatPrice';
import {getNewSortBy} from 'projects/hotelsExtranet/utilities/getNewSortBy';

import * as i18nBlock from 'i18n/hotelsExtranet';

import PaymentsTableRow from './components/PaymentsTableRow/PaymentsTableRow';
import Table, {
    TableBody,
    TableFooter,
    TableFooterCell,
    TableHead,
    TableHeadCell,
    TableRow,
} from 'projects/hotelsExtranet/components/Table/Table';
import Flex from 'components/Flex/Flex';
import Spinner from 'components/Spinner/Spinner';
import SortIcon from 'projects/hotelsExtranet/components/SortIcon/SortIcon';

import cx from './PaymentsTable.scss';

interface IPaymentsTableProps extends IWithClassName {
    items: IExtranetPaymentInfo[];
    total?: IPrice;
    sortBy?: ISortBy<TSortFieldName>;
    withSpinner: boolean;
}

const COLUMNS = [
    {
        title: i18nBlock.transferDate(),
        name: 'eventTime',
        headerCellClassName: cx('date'),
    },
    {
        title: i18nBlock.transferAmount(),
        name: 'sum',
        headerCellClassName: cx('amount'),
    },
    {
        title: i18nBlock.paymentNumber(),
        name: 'bankOrderId',
    },
];

const PaymentsTable: FunctionComponent<IPaymentsTableProps> = ({
    className,
    items,
    total,
    sortBy,
    withSpinner,
}) => {
    const dispatch = useDispatch();

    const handleHeadClick = useCallback(
        (fieldName: TSortFieldName): void => {
            const newSortBy = getNewSortBy<TSortFieldName>(fieldName, sortBy);

            dispatch(setSortField(newSortBy));
        },
        [dispatch, sortBy],
    );

    const header = useMemo(
        () => (
            <TableHead>
                <TableRow>
                    {COLUMNS.map(({title, name, headerCellClassName}) => (
                        <TableHeadCell
                            key={title}
                            className={headerCellClassName}
                            onClick={(): void =>
                                handleHeadClick(name as TSortFieldName)
                            }
                        >
                            {title}
                            <SortIcon
                                columnName={name}
                                sortField={sortBy?.field}
                                sortDirection={sortBy?.sortDirection}
                            />
                        </TableHeadCell>
                    ))}
                </TableRow>
            </TableHead>
        ),
        [sortBy, handleHeadClick],
    );

    const footer = useMemo((): ReactNode => {
        if (withSpinner) {
            return (
                <TableFooter>
                    <TableRow>
                        <TableFooterCell colSpan={9}>
                            <Flex alignItems="center" justifyContent="center">
                                <Spinner size="xxs" />
                                <span className={cx('loadingText')}>
                                    {i18nBlock.loadingOrders()}
                                </span>
                            </Flex>
                        </TableFooterCell>
                    </TableRow>
                </TableFooter>
            );
        }

        if (total) {
            return (
                <TableFooter>
                    <TableRow>
                        <TableFooterCell>{i18nBlock.total()}</TableFooterCell>
                        <TableFooterCell colSpan={2}>
                            {formatPrice(total)}
                        </TableFooterCell>
                    </TableRow>
                </TableFooter>
            );
        }

        return null;
    }, [withSpinner, total]);

    return (
        <Table className={className}>
            {header}
            <TableBody>
                {items.map(item => (
                    <PaymentsTableRow
                        cellClassName={cx('tableCell')}
                        item={item}
                        key={item.bankOrderId}
                    />
                ))}
            </TableBody>
            {footer}
        </Table>
    );
};

export default PaymentsTable;
