import {FC, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import IPrice from 'types/common/price/IPrice';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {TTransactionType} from 'server/api/HotelsExtranetApi/types/IHotelsExtranetPaymentDetailsResponse';

import {fetchPaymentDetails} from 'reducers/hotelsExtranet/paymentsDetails/thunk';

import {allPaymentsDetailsSelector} from 'selectors/hotelsExtranet/allPaymentsDetailsSelector';

import {formatPrice} from 'projects/hotelsExtranet/utilities/formatPrice';
import {CHAR_EM_DASH} from 'utilities/strings/charCodes';

import * as i18nBlock from 'i18n/hotelsExtranet';

import Table, {
    TableBody,
    TableCell,
    TableDateCell,
    TableFooter,
    TableFooterCell,
    TableHead,
    TableHeadCell,
    TablePriceCell,
    TableRow,
} from 'projects/hotelsExtranet/components/Table/Table';
import Spinner from 'components/Spinner/Spinner';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';

import cx from './PaymentDetails.scss';

interface IPaymentDetailsProps {
    paymentBatchId: string;
    total: IPrice;
}

function getTransactionType(transactionType: TTransactionType): string {
    switch (transactionType) {
        case 'BTTT_PAYMENT':
            return i18nBlock.payment();
        case 'BTTT_REFUND':
            return i18nBlock.refund();
        default:
            return transactionType;
    }
}

const PaymentDetails: FC<IPaymentDetailsProps> = ({paymentBatchId}) => {
    const dispatch = useDispatch();
    const allPaymentsDetails = useSelector(allPaymentsDetailsSelector);

    const paymentDetails = useMemo(() => {
        return allPaymentsDetails[paymentBatchId];
    }, [allPaymentsDetails, paymentBatchId]);

    useEffect(() => {
        if (!paymentDetails) {
            dispatch(fetchPaymentDetails({paymentBatchId}));
        }
    }, [paymentDetails, dispatch, paymentBatchId]);

    if (!paymentDetails || paymentDetails.status === EAsyncStatus.LOADING) {
        return (
            <Flex
                className={cx('spinnerWrap')}
                alignItems="center"
                justifyContent="center"
                inline
                between={3}
            >
                <Spinner size="xxs" />
                <span className={cx('loadingText')}>
                    {i18nBlock.loadingOrders()}
                </span>
            </Flex>
        );
    }

    if (
        paymentDetails.status === EAsyncStatus.ERROR ||
        !paymentDetails.response
    ) {
        return (
            <Text className={cx('error')} tag="div" color="alert">
                {i18nBlock.failedToLoadPayments()}
            </Text>
        );
    }

    const {sum, result} = paymentDetails.response;

    return (
        <Table>
            <TableHead>
                <TableRow>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.yandexId()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.guestName()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.bookingDate()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.checkIn()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.checkOut()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.paymentType()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.paidByGuest()}
                    </TableHeadCell>
                    <TableHeadCell noSideBorders>
                        {i18nBlock.listedToHotel()}
                    </TableHeadCell>
                </TableRow>
            </TableHead>
            <TableBody>
                {result.map(
                    ({
                        orderPrettyId,
                        firstGuest,
                        orderCreatedAt,
                        checkInDate,
                        checkOutDate,
                        priceBreakdown,
                        transactionType,
                    }) => (
                        <TableRow key={orderPrettyId}>
                            <TableCell noSideBorders>{orderPrettyId}</TableCell>
                            <TableCell noSideBorders>
                                {firstGuest
                                    ? `${firstGuest.firstName} ${firstGuest.lastName}`
                                    : CHAR_EM_DASH}
                            </TableCell>
                            <TableDateCell
                                noSideBorders
                                date={orderCreatedAt}
                            />
                            <TableDateCell noSideBorders date={checkInDate} />
                            <TableDateCell noSideBorders date={checkOutDate} />
                            <TableCell noSideBorders>
                                {getTransactionType(transactionType)}
                            </TableCell>
                            <TablePriceCell
                                noSideBorders
                                price={priceBreakdown.fiscalPrice}
                            />
                            <TablePriceCell
                                noSideBorders
                                price={priceBreakdown.partner}
                            />
                        </TableRow>
                    ),
                )}
            </TableBody>
            <TableFooter>
                <TableRow>
                    <TableFooterCell className={cx('total')} colSpan={7}>
                        {i18nBlock.paymentSum()}
                    </TableFooterCell>
                    <TableFooterCell>{formatPrice(sum)}</TableFooterCell>
                </TableRow>
            </TableFooter>
        </Table>
    );
};

export default PaymentDetails;
