import {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {
    generatePath,
    matchPath,
    useHistory,
    useLocation,
} from 'react-router-dom';

import {URLs} from 'constants/urls';

import {IExtranetHotel} from 'server/api/HotelsExtranetApi/types/IHotelsListResponse';
import EAsyncStatus from 'types/common/EAsyncStatus';

import {
    getHotelsList,
    setSelectedHotel,
} from 'reducers/hotelsExtranet/hotelsList/thunk';

import {hotelsListSelector} from '../../../redux/selectors/hotelsExtranet/hotelsListSelector';

import {useUserInfo} from 'utilities/hooks/useUserInfo';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import {useAddFormQueryParams} from './useAddFormQueryParams';

interface IUseHotelsList {
    hotels?: IExtranetHotel[];
    currentHotel?: IExtranetHotel;
    isPending: boolean;
}

interface IHotelInfo {
    hotelCode: string;
    partnerId: string;
}

const useHotelsList = (
    canLoad: boolean,
    canRedirect: boolean,
): IUseHotelsList => {
    const dispatch = useDispatch();
    const {pathname} = useLocation();
    const history = useHistory();
    const userInfo = useUserInfo();
    const {status, response, currentHotel} = useSelector(hotelsListSelector);
    const {hotels} = response || {};
    const atAddHotelPage = pathname === URLs.hotelsExtranetAddHotelPage;
    const {fromAuth} = useAddFormQueryParams();

    const redirectToFirstHotel = useCallback(() => {
        if (hotels?.length) {
            const [firstHotel] = hotels;
            const {hotelCode, partnerId} = firstHotel;
            const newPath = generatePath(URLs.hotelsExtranetIndexPage, {
                hotelCode,
                partnerId,
            });

            dispatch(setSelectedHotel(firstHotel));
            history.replace(newPath);
        }
    }, [dispatch, hotels, history]);

    useEffect(() => {
        if (!canLoad || !isAuthUser(userInfo)) {
            return;
        }

        if (!status) {
            const hotelIdentifier = tryFindHotelSlug(pathname);

            dispatch(getHotelsList(hotelIdentifier));
        }
    }, [dispatch, status, pathname, canLoad, userInfo]);

    useEffect(() => {
        if (!canRedirect) {
            return;
        }

        if (atAddHotelPage) {
            if (fromAuth && hotels?.length) {
                redirectToFirstHotel();
            }
        } else if (!currentHotel && status === EAsyncStatus.SUCCESS) {
            if (hotels?.length) {
                redirectToFirstHotel();
            } else {
                history.replace(URLs.hotelsExtranetAddHotelPage);
            }
        }
    }, [
        currentHotel,
        history,
        status,
        hotels,
        canRedirect,
        fromAuth,
        atAddHotelPage,
        dispatch,
        redirectToFirstHotel,
    ]);

    useEffect(() => {
        if (!atAddHotelPage && canRedirect && !isAuthUser(userInfo)) {
            history.push(URLs.hotelsExtranetAddHotelPage);
        }
    }, [userInfo, history, canRedirect, atAddHotelPage]);

    return useMemo(
        () => ({
            hotels,
            currentHotel,
            isPending: !status || status === EAsyncStatus.LOADING,
        }),
        [hotels, currentHotel, status],
    );
};

const URLsWithHotelSlug = [
    URLs.hotelsExtranetIndexPage,
    URLs.hotelsExtranetOrdersPage,
    URLs.hotelsExtranetPaymentsPage,
];

function tryFindHotelSlug(path: string): IExtranetHotel | undefined {
    return URLsWithHotelSlug.map(template => {
        const match = matchPath<IHotelInfo>(path, {
            path: template,
            exact: true,
        });

        return match?.params;
    }).find(Boolean);
}

export default useHotelsList;
