import React, {useEffect, useMemo} from 'react';
import {useLocation} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';
import {NAVIGATION_INDEX_TABS_PROJECTS_ORDER} from 'constants/navigation';

import {previousSearchFormFillForm} from 'reducers/common/previousSearchForm/actions';

import {aviaPlatformSelector} from 'selectors/avia/platform/aviaPlatformSelector';
import {tabBarEnabledSelector} from 'selectors/common/tabBarEnabledSelector';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {internalUrl} from 'utilities/url';
import getProjectName from 'server/utilities/getProjectName';
import {useEnvironment} from 'utilities/hooks/useEnvironment';
import usePreloadChunks from 'utilities/hooks/usePreloadChunks';
import getMetaTitleText from 'projects/index/IndexApp/utilities/getMetaTitleText';
import getMetaDescriptionText from 'projects/index/IndexApp/utilities/getMetaDescriptionText';
import getMetaOgDescriptionText from 'projects/index/IndexApp/utilities/getMetaOgDescriptionText';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useWhiteLabelConfig} from 'utilities/hooks/useWhiteLabelConfig';

import TabBar from 'components/TabBar/TabBar';
import DocumentMeta from 'components/DocumentMeta/DocumentMeta';
import IndexLayout from 'components/Layouts/IndexLayout/IndexLayout';
import MriaPopup from 'projects/index/IndexApp/components/MriaPopup/MriaPopup';
import MriaBannerLine from 'projects/index/IndexApp/components/MriaBannerLine/MriaBannerLine';
import Footer from 'components/Layouts/IndexLayout/components/Footer/Footer';
import Header from 'components/Layouts/IndexLayout/components/Header/Header';
import SocialSharingMeta from 'components/SocialSharingMeta/SocialSharingMeta';
import IndexAviaLoadable from './components/IndexTabs/components/IndexAviaLoadable';
import IndexBusesLoadable from './components/IndexTabs/components/IndexBusesLoadable';
import IndexTrainsLoadable from './components/IndexTabs/components/IndexTrainsLoadable';
import IndexHotelsLoadable from './components/IndexTabs/components/IndexHotelsLoadable';
import IndexToursLoadable from './components/IndexTabs/components/IndexToursLoadable';
import {AviaAndroidAppHeader} from 'projects/avia/components/AviaAndroidAppHeader/AviaAndroidAppHeader';
import IndexTabs from './components/IndexTabs/IndexTabs';

import {aviaLoggingBrowserProvider} from 'serviceProvider/avia/logging/aviaLoggingBrowserProvider';

const CHUNKS_TO_PRELOAD = [
    IndexAviaLoadable,
    IndexTrainsLoadable,
    IndexHotelsLoadable,
    IndexBusesLoadable,
    IndexToursLoadable,
];

const IndexApp: React.FC = () => {
    const dispatch = useDispatch();
    const location = useLocation();
    const {tld} = useEnvironment();

    const platform = useSelector(aviaPlatformSelector);
    const tabBarEnabled = useSelector(tabBarEnabledSelector);
    const {isDesktop, isMobile} = useDeviceType();
    const {isWhiteLabel} = useWhiteLabelConfig();

    const tabsList = NAVIGATION_INDEX_TABS_PROJECTS_ORDER;

    const activeTabId = useMemo(() => {
        const projectByPathname = getProjectName(location.pathname);
        const [firstProjectId] = tabsList;

        return tabsList.reduce((validProjectId, currentProjectId) => {
            const currentProjectPathname = getProjectName(
                URLs[currentProjectId],
            );

            if (
                currentProjectPathname &&
                currentProjectPathname === projectByPathname
            ) {
                return currentProjectId;
            }

            return validProjectId;
        }, firstProjectId);
    }, [location.pathname, tabsList]);

    const helmetBlock = useMemo(() => {
        const projectByPathname = getProjectName(location.pathname);

        if (projectByPathname !== EProjectName.INDEX) {
            const canonicalUrl = internalUrl(URLs[activeTabId], null, {
                withOrigin: true,
                tld,
            });

            return (
                <>
                    <DocumentMeta
                        title={getMetaTitleText(activeTabId)}
                        description={getMetaDescriptionText(activeTabId)}
                        canonicalUrl={canonicalUrl}
                    />

                    <SocialSharingMeta
                        url={canonicalUrl}
                        title={getMetaTitleText(activeTabId)}
                        description={getMetaOgDescriptionText(activeTabId)}
                    />
                </>
            );
        }

        return (
            <DocumentMeta
                extra={[
                    {
                        name: 'yandex-verification',
                        content: 'ef25e3a439f4be37',
                    },
                    {
                        name: 'google-site-verification',
                        content: 'F5Y5yEn18DN1IK65G7p0LQZAOPjO9ZAN4l2IvBiGroU',
                    },
                ]}
                canonicalUrl={internalUrl(URLs[EProjectName.INDEX], null, {
                    withOrigin: true,
                    tld,
                })}
            />
        );
    }, [activeTabId, location.pathname, tld]);

    const headerBlock = useMemo(() => {
        const {isWeb, isAndroid, version} = platform;

        if (isWeb) {
            const projectByPathname = getProjectName(location.pathname);

            return (
                <Header
                    project={projectByPathname}
                    withNavCaptions={isDesktop}
                    showFavoritesLink={!tabBarEnabled}
                />
            );
        }

        if (isAndroid && version && parseFloat(version) >= 2.0) {
            return <AviaAndroidAppHeader />;
        }

        return null;
    }, [isDesktop, location.pathname, platform, tabBarEnabled]);

    const footerBlock = useMemo(() => {
        if (!platform.isWeb) {
            return null;
        }

        return (
            <>
                <Footer
                    project={
                        isWhiteLabel ? EProjectName.WHITE_LABEL : activeTabId
                    }
                />
                <MriaPopup />
                <MriaBannerLine />
            </>
        );
    }, [activeTabId, isWhiteLabel, platform.isWeb]);

    const tabsBlock = useMemo(() => {
        const query = getQueryByLocation(location);

        return (
            <IndexTabs
                query={query}
                tabs={tabsList}
                activeTabId={activeTabId}
                isWeb={platform.isWeb}
            />
        );
    }, [activeTabId, location, platform.isWeb, tabsList]);

    useEffect(() => {
        dispatch(previousSearchFormFillForm());

        aviaLoggingBrowserProvider.landingVisit();
    }, [dispatch]);

    usePreloadChunks(CHUNKS_TO_PRELOAD);

    return (
        <>
            {helmetBlock}

            <IndexLayout
                header={headerBlock}
                footer={footerBlock}
                tabBar={isMobile && tabBarEnabled && <TabBar />}
            >
                {tabsBlock}
            </IndexLayout>
        </>
    );
};

export default IndexApp;
