import React, {useMemo} from 'react';
import times from 'lodash/times';

import ICrossLinksGalleryItem from 'projects/index/IndexApp/components/CrossLinksGallery/types/ICrossLinksGalleryItem';
import {IWithClassName} from 'types/withClassName';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getItemType from 'projects/index/IndexApp/components/CrossLinksGallery/utilities/getItemType';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/components-FeaturedDirections';

import CrossLinkItem from 'projects/index/IndexApp/components/CrossLinksGallery/components/CrossLinkItem/CrossLinkItem';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import CrossLinkItemSkeleton from 'projects/index/IndexApp/components/CrossLinksGallery/components/CrossLinkItem/components/CrossLinkItemSkeleton/CrossLinkItemSkeleton';
import Link from 'components/Link/Link';
import Text from 'components/Text/Text';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import InNewTabIcon from 'icons/16/InNewTab';
import CrossLinksSchemaMarkup from './components/CrossLinksSchemaMarkup/CrossLinksSchemaMarkup';

import cx from './CrossLinksGallery.scss';

interface ICrossLinksGalleryProps extends IWithClassName, IWithQaAttributes {
    title: React.ReactNode;
    items: ICrossLinksGalleryItem[];
    moreUrl?: string;
    /**
     * Количество карточек крупных
     * чтобы выглядело нормально - нужно число кратное 4
     * @default 8
     */
    cardCount?: number;
    loading?: boolean;
    nonce: string;

    /**
     * Текст для markup разметки
     */
    markupText: string;

    /**
     * Цель отправляемая по клику на элемент
     */
    itemClickMetrikaGoal?: TMetrikaGoal;
}

const CrossLinksGallery: React.FC<ICrossLinksGalleryProps> = props => {
    const {
        className,
        title,
        items,
        cardCount = 8,
        loading = false,
        moreUrl,
        nonce,
        markupText,
        itemClickMetrikaGoal,
    } = props;

    const deviceType = useDeviceType();

    const rootQa = getQa(props);

    const itemsNode = useMemo(() => {
        const itemsNodes = loading
            ? times(cardCount * 2).map(index => (
                  <CrossLinkItemSkeleton
                      className={cx('item')}
                      key={index}
                      type={getItemType({deviceType, index, cardCount})}
                      {...prepareQaAttributes({
                          parent: rootQa,
                          current: 'itemSkeleton',
                          key: index,
                      })}
                  />
              ))
            : items.map((item, index) => (
                  <CrossLinkItem
                      key={item.id}
                      className={cx('item')}
                      type={getItemType({deviceType, index, cardCount})}
                      item={item}
                      itemClickMetrikaGoal={itemClickMetrikaGoal}
                      {...prepareQaAttributes({
                          parent: rootQa,
                          current: 'item',
                          key: index,
                      })}
                  />
              ));

        const content = <div className={cx('items')}>{itemsNodes}</div>;

        if (deviceType.isDesktop) {
            return content;
        }

        return (
            <HorizontalScroller
                className={cx('scroller')}
                scrollableClassName={cx('scrollable')}
                offset={4}
            >
                {content}
            </HorizontalScroller>
        );
    }, [cardCount, deviceType, itemClickMetrikaGoal, items, loading, rootQa]);

    return (
        <div
            className={cx(className, 'root', deviceMods('root', deviceType))}
            {...prepareQaAttributes(rootQa)}
        >
            <Text
                className={cx('title')}
                tag="h2"
                size={deviceType.isDesktop ? 'xxl' : 'xl'}
                align="center"
                {...prepareQaAttributes({parent: props, current: 'title'})}
            >
                {title}
            </Text>
            {itemsNode}
            {moreUrl && (
                <div className={cx('morePopular')}>
                    <Link
                        theme="black"
                        target="_blank"
                        url={moreUrl}
                        {...prepareQaAttributes({
                            parent: rootQa,
                            current: 'moreLink',
                        })}
                    >
                        <TextWithIcon
                            size="m"
                            text={i18nBlock.morePopular()}
                            iconRight={InNewTabIcon}
                        />
                    </Link>
                </div>
            )}
            <CrossLinksSchemaMarkup
                nonce={nonce}
                markupText={markupText}
                items={items}
            />
        </div>
    );
};

export default CrossLinksGallery;
