import React, {useCallback, useMemo} from 'react';

import ICrossLinksGalleryItem from 'projects/index/IndexApp/components/CrossLinksGallery/types/ICrossLinksGalleryItem';
import {IWithClassName} from 'types/withClassName';
import ECrossLinkItemType from 'projects/index/IndexApp/components/CrossLinksGallery/components/CrossLinkItem/types/ECrossLinkItemType';
import {TMetrikaGoal} from 'utilities/metrika/types/goals/all';

import getImageSizeByType from 'projects/index/IndexApp/components/CrossLinksGallery/components/CrossLinkItem/utilities/getImageSizeByType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';

import CrossLinkImage from 'projects/index/IndexApp/components/CrossLinksGallery/components/CrossLinkImage/CrossLinkImage';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import Link from 'components/Link/Link';

import cx from './CrossLinkItem.scss';

interface ICrossLinkItemProps extends IWithClassName, IWithQaAttributes {
    type: ECrossLinkItemType;
    item: ICrossLinksGalleryItem;
    loading?: boolean;
    itemClickMetrikaGoal?: TMetrikaGoal;
    onClick?: React.MouseEventHandler;
}

const CrossLinkItem: React.FC<ICrossLinkItemProps> = props => {
    const {
        className,
        type,
        item,
        loading = false,
        itemClickMetrikaGoal,
        onClick,
    } = props;

    const imageSize = useMemo(() => getImageSizeByType(type), [type]);

    const handleClick: React.MouseEventHandler = useCallback(
        e => {
            if (itemClickMetrikaGoal) {
                reachGoal(itemClickMetrikaGoal);
            }

            onClick?.(e);
        },
        [itemClickMetrikaGoal, onClick],
    );

    return (
        <div
            className={cx(className, 'root', `root_${type}`)}
            {...prepareQaAttributes(props)}
            onClick={handleClick}
        >
            <CrossLinkImage
                className={cx('image')}
                src={item.image}
                size={imageSize}
                loading={loading}
                {...prepareQaAttributes({parent: props, current: 'image'})}
            />
            <Flex flexDirection="column" between={1}>
                <Text
                    size="l"
                    weight="bold"
                    {...prepareQaAttributes({parent: props, current: 'title'})}
                >
                    {item.title}
                </Text>
                {item.price && (
                    <Text size="m">
                        <Price
                            isFrom
                            {...item.price}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'price',
                            })}
                        />
                    </Text>
                )}
            </Flex>
            {item.url && (
                <Link
                    className={cx('link')}
                    url={item.url}
                    target="_blank"
                    {...prepareQaAttributes({parent: props, current: 'link'})}
                />
            )}
        </div>
    );
};

export default CrossLinkItem;
