import React, {useEffect, useMemo, useRef, useState} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {ParsedQuery} from 'query-string';

import {EProjectName} from 'constants/common';

import {ServerDataFetcherBag} from 'server/redux/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {IAviaSuggest} from 'types/avia/suggests/IAviaSuggest';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {
    isSearchFormFieldName,
    TSearchFormErrors,
} from 'components/SearchForm/types';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import searchSuggestsReducer from 'reducers/avia/searchSuggests/reducer';
import searchResultsReducer from 'reducers/avia/search/reducer';
import hotelsPersonalizationReducer from 'reducers/hotels/personalization/reducer';
import personalizationReducer from 'reducers/avia/personalization/reducer';
import calendarPricesReducer from 'reducers/avia/aviaCalendarPrices/reducer';
import aviaLogMetricsReducer from 'reducers/avia/aviaLogMetrics/reducer';
import {FeaturesState} from 'reducers/common/commonReducerTypes';
import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import nonceSelector from 'selectors/common/nonceSelector';
import crossLinksSelector from 'selectors/common/crossLinksSelector';

import pricesSaga from 'sagas/avia/aviaCalendarPrices/aviaCalendarPricesSaga';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {reachGoal} from 'utilities/metrika';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import prepareCrossLinksForGallery from 'utilities/prepareCrossLinksForGallery';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {useExperiments} from 'utilities/hooks/useExperiments';

import * as i18nSearchForm from 'i18n/avia-SearchForm';
import * as i18nAviaRecipe from 'i18n/avia-AviaRecipe';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import SearchForm from 'projects/avia/components/SearchForm/SearchForm';
import AviaAdvantages from 'projects/avia/components/AviaAdvantages/AviaAdvantages';
import SearchPresets from './components/SearchPresets/SearchPresets';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';
import IndexSubscriptionForm from '../../components/IndexSubscriptionForm/IndexSubscriptionForm';
import Text from 'components/Text/Text';
import CrossLinksGallery from 'projects/index/IndexApp/components/CrossLinksGallery/CrossLinksGallery';
import AviaFAQ from 'projects/index/IndexApp/components/IndexAvia/components/AviaFAQ/AviaFAQ';

import prefetchCrossLinks from 'server/redux/common/fetchCrossLinks';
import {prefillAviaSearchForm} from 'server/redux/avia/prefillSearchForm';

import cx from './IndexAvia.scss';

const ROOT_QA = 'indexAviaPage';

const unknownFromError = i18nSearchForm.validateUnknownFromDirection();
const unknownToError = i18nSearchForm.validateUnknownToDirection();

export interface IIndexAviaOwnProps {
    query: ParsedQuery;
}

export interface IIndexAviaStatePropsContainer {
    features: FeaturesState;
}

async function prefetchData(bag: ServerDataFetcherBag): Promise<void> {
    await Promise.all([
        prefillAviaSearchForm(bag),
        prefetchCrossLinks(bag, EProjectName.AVIA),
    ]);
}

const IndexAvia: React.FC<IIndexAviaOwnProps> = props => {
    const {query = {}} = props;

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const nonce = useSelector(nonceSelector);

    const crossLinksState = useSelector(crossLinksSelector);
    const {
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = crossLinksState;
    const galleryItems = useMemo(() => {
        return prepareCrossLinksForGallery(crossLinksState, EProjectName.AVIA);
    }, [crossLinksState]);

    const {rebrandingSubscriptions} = useExperiments();

    const needToRender = useServerDataFetcher([prefetchData]);

    const searchFormFieldsRef = useRef<IFormFieldsRef>();
    const prevFormErrors = useRef<TSearchFormErrors>({});

    const onValidationFailed = useImmutableCallback(
        (formErrors: TSearchFormErrors) => {
            if (
                formErrors.from?.includes(unknownFromError) &&
                !prevFormErrors.current.from?.includes(unknownFromError)
            ) {
                reachGoal(EIndexGoal.AVIA_MORDA_SEARCH_FORM_UNKNOWN_FROM);
            }

            if (
                formErrors.to?.includes(unknownToError) &&
                !prevFormErrors.current.to?.includes(unknownToError)
            ) {
                reachGoal(EIndexGoal.AVIA_MORDA_SEARCH_FORM_UNKNOWN_TO);
            }

            prevFormErrors.current = formErrors;
        },
    );

    const onResetCalendar = useReachGoal(
        EIndexGoal.AVIA_MORDA_SEARCH_FORM_RESET_DATE,
    );

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (project !== EProjectName.AVIA || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.AVIA));
        }
    }, [dispatch, isRecipesFetched, project]);

    const onChangeFromPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IAviaSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.AVIA_MORDA_SEARCH_FORM_CLEAR_FROM);
            }
        },
    );

    const onChangeToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IAviaSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.AVIA_MORDA_SEARCH_FORM_CLEAR_TO);
            }
        },
    );

    const [customToPoint, setCustomToPoint] = useState<
        ISuggestValue<IAviaSuggest> | undefined
    >(undefined);

    const setToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IAviaSuggest>) => {
            setCustomToPoint(fieldValue);
        },
    );

    const crossLinkGalleryTitle = useMemo(() => {
        return insertJSXIntoKey(i18nAviaRecipe.galleryTitle)({
            start: (
                <Text weight="fat" size="inherit">
                    {i18nAviaRecipe.galleryTitleStart()}
                </Text>
            ),
        });
    }, []);

    if (!needToRender) {
        return null;
    }

    const {focus} = query;
    const autoFocusFieldName = isSearchFormFieldName(focus) ? focus : undefined;

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.AVIA;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), {
                root_rebrandingSubscriptions: rebrandingSubscriptions,
            })}
        >
            <SearchForm
                className={cx('searchForm')}
                needToSetFromSuggestByGeoPosition
                useFiltersFromUrl
                query={query}
                autoFocusFieldName={autoFocusFieldName}
                fieldsRef={searchFormFieldsRef}
                onValidationFailed={onValidationFailed}
                onResetCalendar={onResetCalendar}
                onChangeFromPoint={onChangeFromPoint}
                onChangeToPoint={onChangeToPoint}
                customToPoint={customToPoint}
            />
            <SearchPresets
                className={cx('searchPresets')}
                searchFormFieldsRef={searchFormFieldsRef}
                setToPoint={setToPoint}
            />

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            <AviaAdvantages
                className={cx('advantages')}
                {...prepareQaAttributes(ROOT_QA)}
            />

            <IndexSubscriptionForm
                className={cx('subscribe')}
                vertical={ESubscriptionVerticalName.Avia}
            />

            {renderCrossaleBlock && (
                <CrossLinksGallery
                    className={cx('recipes')}
                    title={crossLinkGalleryTitle}
                    items={galleryItems}
                    loading={isRecipesLoading}
                    nonce={nonce}
                    markupText={i18nAviaRecipe.markupDashText()}
                    itemClickMetrikaGoal={ECommonGoal.RECIPE_LINK}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'crossLinksGallery',
                    })}
                />
            )}

            <AviaFAQ className={cx('texts')} />
        </div>
    );
};

export default withReducers<IIndexAviaOwnProps>([
    [ELoadableReducer.AVIA_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.AVIA_SEARCH, searchResultsReducer],
    [ELoadableReducer.AVIA_CALENDAR_PRICES, calendarPricesReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
    [ELoadableReducer.AVIA_LOG_METRICS, aviaLogMetricsReducer],
    [ELoadableReducer.HOTELS_PERSONALIZATION, hotelsPersonalizationReducer],
    [ELoadableReducer.AVIA_PERSONALIZATION, personalizationReducer],
])(withSaga<IIndexAviaOwnProps>([pricesSaga])(IndexAvia));
