import React, {ReactNode, useCallback, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';

import {useDeviceType} from 'utilities/hooks/useDeviceType';

import * as i18n from 'i18n/avia-indexTexts';

import CardWithDeviceLayout from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Swipeable, {
    ISwipeableRenderItemParams,
} from 'components/Swipeable/Swipeable';
import Fair3DImage from 'icons/raster-illustrations/L/Fair3D';

import cx from './AviaFAQ.scss';

type TSwipeableRenderItemParams = Omit<
    ISwipeableRenderItemParams<{
        title: string;
        description: string;
    }>,
    'isCurrent'
>;

interface IAviaFAQProps extends IWithClassName {}

const AviaFAQ: React.FC<IAviaFAQProps> = props => {
    const {className} = props;
    const deviceType = useDeviceType();

    const items = useMemo(() => {
        return [
            {
                title: i18n.howItWorksTitle(),
                description: i18n.howItWorksDescription(),
            },
            {
                title: i18n.howToFindProfitTitle(),
                description: i18n.howToFindProfitDescription(),
            },
        ];
    }, []);

    const renderItem = useCallback(
        ({data, index}: TSwipeableRenderItemParams): ReactNode => (
            <Flex key={index} flexDirection="column" between={3}>
                <Text
                    size="l"
                    weight="bold"
                    align={deviceType.isDesktop ? undefined : 'center'}
                >
                    {data.title}
                </Text>
                <Text align={deviceType.isDesktop ? undefined : 'center'}>
                    {data.description}
                </Text>
            </Flex>
        ),
        [deviceType],
    );

    const itemNodes = useMemo(() => {
        if (deviceType.isMobile) {
            return (
                <Swipeable
                    className={cx('swipeable')}
                    sideThreshold={0.1}
                    renderItem={renderItem}
                    itemsData={items}
                    itemVerticalAlignment="start"
                    showNav
                />
            );
        }

        return (
            <Flex flexDirection="column" between={8}>
                {items.map((data, index) => renderItem({data, index}))}
            </Flex>
        );
    }, [deviceType, items, renderItem]);

    return (
        <CardWithDeviceLayout
            className={className}
            shadow="none"
            background="grey"
        >
            <Flex
                flexDirection={deviceType.isDesktop ? 'row' : 'column'}
                inline={deviceType.isDesktop}
                between={deviceType.isDesktop ? 8 : 4}
                alignItems="center"
                nowrap
            >
                <Fair3DImage className={cx('image')} />
                {itemNodes}
            </Flex>
        </CardWithDeviceLayout>
    );
};

export default AviaFAQ;
