import React, {MutableRefObject, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import EPersonalizedItemType from 'server/api/PersonalizationApi/types/EPersonalizedItemType';
import {IAviaSuggest} from 'types/avia/suggests/IAviaSuggest';

import {resetAviaSearchHistoryAction} from 'reducers/avia/personalization/actions';

import {hotelsPersonalizationSelector} from 'selectors/hotels/personalization/hotelsPersonalizationSelector';
import {aviaPersonalizationSearchHistory} from 'selectors/avia/personalization/aviaPersonalizationSearchHistory';

import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import formatSearchHistoryItemsToSearchFormItem from '../../../../utilities/formatSearchHistoryItemsToSearchFormItem';
import convertSearchHistoryToAviaMinPriceBatchSearchParams from './utilities/convertSearchHistoryToAviaMinPriceBatchSearchParams';
import findPrice from './utilities/findPrice';
import {useExperiments} from 'utilities/hooks/useExperiments';

import Flex from 'components/Flex/Flex';
import PreviousSearchPreset from './components/PreviousSearchPreset/PreviousSearchPreset';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import CrossSalePreset from './components/CrossSalePreset/CrossSalePreset';
import AnywherePreset from './components/AnywherePreset/AnywherePreset';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';

import useMinPriceForSearchPresets from './hooks/useMinPriceForSearchPresets';

interface ISearchPresets extends IWithClassName, IWithQaAttributes {
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
    setToPoint(fieldValue: ISuggestValue<IAviaSuggest>): void;
}

const SearchPresets: React.FC<ISearchPresets> = props => {
    const {className, searchFormFieldsRef, setToPoint} = props;

    const rootQa = getQa(props);
    const deviceType = useDeviceType();

    const {aviaSearchToAnywherePage} = useExperiments();

    const dispatch = useDispatch();
    const {suggests} = useSelector(hotelsPersonalizationSelector);
    const {searchHistory} = useSelector(aviaPersonalizationSearchHistory);

    const {fetch, prices, isLoading} = useMinPriceForSearchPresets();

    useEffect(() => {
        const request =
            convertSearchHistoryToAviaMinPriceBatchSearchParams(searchHistory);

        if (!request.minRequests.length) {
            return;
        }

        fetch(request);

        return (): void => {
            dispatch(resetAviaSearchHistoryAction());
        };
    }, [dispatch, searchHistory, fetch]);

    const crossSaleItem = suggests.items.find(
        item =>
            item.type === EPersonalizedItemType.CROSS_SALE ||
            item.type === EPersonalizedItemType.CROSS_SEARCH,
    );

    const itemsCount =
        (aviaSearchToAnywherePage ? 1 : 0) +
        (crossSaleItem ? 1 : 0) +
        searchHistory.items.length;

    const content = useMemo(() => {
        if (!itemsCount) {
            return null;
        }

        const width = deviceType.isMobile && itemsCount === 1 ? 'max' : 'auto';

        return (
            <Flex
                className={className}
                between={2}
                inline
                nowrap
                {...prepareQaAttributes(rootQa)}
            >
                {aviaSearchToAnywherePage && (
                    <AnywherePreset
                        searchFormFieldsRef={searchFormFieldsRef}
                        setToPoint={setToPoint}
                        index={0}
                    />
                )}

                {crossSaleItem && (
                    <CrossSalePreset
                        item={crossSaleItem}
                        width={width}
                        index={1}
                    />
                )}

                {searchHistory.items.map((item, index) => {
                    const price = findPrice(prices, item);

                    return (
                        <PreviousSearchPreset
                            key={index}
                            item={formatSearchHistoryItemsToSearchFormItem(
                                item,
                            )}
                            searchFormFieldsRef={searchFormFieldsRef}
                            width={width}
                            isLoading={isLoading}
                            price={price}
                            index={2 + index}
                        />
                    );
                })}
            </Flex>
        );
    }, [
        itemsCount,
        deviceType.isMobile,
        className,
        rootQa,
        aviaSearchToAnywherePage,
        searchFormFieldsRef,
        setToPoint,
        crossSaleItem,
        searchHistory.items,
        prices,
        isLoading,
    ]);

    if (!content) {
        return null;
    }

    if (deviceType.isDesktop || itemsCount === 1) {
        return content;
    }

    return <HorizontalScroller>{content}</HorizontalScroller>;
};

export default React.memo(SearchPresets);
