import React, {MutableRefObject, useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {EHistoryControlType} from 'server/loggers/avia/AviaActionLog/types/EHistoryControlType';
import {IAviaSearchFormValues} from 'projects/avia/components/SearchForm/types';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {EAviaGoal} from 'utilities/metrika/types/goals/avia';
import {IAviaMinPriceSearchResult} from 'server/api/AviaPriceIndexApi/types/IAviaMinPriceBatchSearchApiResponse';

import {applyPreviousSearchAction} from 'reducers/avia/searchForm/thunk/applyPreviousSearchAction';
import {setHistoryControlUsedAction} from 'reducers/avia/aviaLogMetrics/actions';
import {resetSearch} from 'reducers/avia/search/results/actions';

import {
    isOutdated,
    preparePreviousSearchData,
} from 'projects/avia/lib/previousSearches';
import {reachGoal} from 'utilities/metrika';
import {getSearchFormUrl} from 'projects/avia/components/SearchForm/utilities/getSearchFormUrl';

import * as i18n from 'i18n/common';

import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import SearchPreset from 'components/SearchPreset/SearchPreset';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import Price from 'components/Price/Price';

import cx from './PreviousSearchPreset.scss';

interface IPreviousSearchPresetProps {
    item: IAviaSearchFormValues;
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
    width: 'auto' | 'max';
    isLoading: boolean;
    price: IAviaMinPriceSearchResult | undefined;
    index: number;
}

const PreviousSearchPreset: React.FC<IPreviousSearchPresetProps> = props => {
    const {item, searchFormFieldsRef, width, isLoading, price, index} = props;
    const history = useHistory();
    const dispatch = useDispatch();

    const handleClick = useCallback(() => {
        if (isOutdated(item)) {
            searchFormFieldsRef.current?.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        } else {
            const {
                from,
                to,
                startDate,
                endDate,
                adults,
                children,
                infants,
                aviaClass: klass,
            } = item;
            const {url} = getSearchFormUrl({
                fromKey:
                    from.selectedValue === false
                        ? ''
                        : from.selectedValue.pointKey,
                toKey:
                    to.selectedValue === false ? '' : to.selectedValue.pointKey,
                dateForward: startDate || '',
                dateBackward: endDate || '',
                adults,
                children,
                infants,
                klass,
            });

            if (url) {
                history?.push(url);
            }
        }

        dispatch(resetSearch());
        dispatch(
            setHistoryControlUsedAction({
                type: EHistoryControlType.PRESET,
            }),
        );
        dispatch(applyPreviousSearchAction(item));

        reachGoal(EAviaGoal.PREVIOUS_SEARCHES_PRESET_CLICK);
    }, [item, dispatch, searchFormFieldsRef, history]);

    const {title, date} = preparePreviousSearchData(item);

    const secondRow = useMemo(() => {
        return date || i18n.selectDates();
    }, [date]);

    const priceNode = useMemo(() => {
        if (isLoading) {
            return (
                <TextSkeleton
                    className={cx('priceSkeleton')}
                    size="m"
                    withAnimation
                />
            );
        }

        const minPrice = price?.minPrice;

        if (!minPrice) {
            return;
        }

        return (
            <Price
                value={minPrice.value}
                currency={minPrice.currency}
                isRound
            />
        );
    }, [price, isLoading]);

    return (
        <SearchPreset
            title={title}
            secondRow={secondRow}
            sideTitle={priceNode}
            width={width}
            theme="blue"
            onClick={handleClick}
            index={index}
        />
    );
};

export default React.memo(PreviousSearchPreset);
