import React, {useEffect, useMemo} from 'react';
import {useLocation} from 'react-router-dom';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';

import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {EBusesGoal} from 'utilities/metrika/types/goals/buses';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import contextReducer from 'reducers/buses/context/reducer';
import calendarReducer from 'reducers/buses/calendar/reducer';
import searchFormReducer from 'reducers/buses/searchForm/reducer';
import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import searchSuggestsReducer from 'reducers/buses/searchSuggests/reducer';
import previousSearchesReducer from 'reducers/buses/previousSearches/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import crossLinksSelector from 'selectors/common/crossLinksSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import requestPreviousSearchesSaga from 'sagas/index/buses/previousSearches/requestPreviousSearchesSaga';

import {reachGoal} from 'utilities/metrika';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import prefetchData from './utilities/prefetchData';
import getIndexQuery from './utilities/getIndexQuery';
import prepareCrossLinksForGallery from 'utilities/prepareCrossLinksForGallery';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBusesIndexBlock from 'i18n/buses-indexPage';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import SearchForm from 'projects/buses/components/SearchForm/SearchForm';
import BusesAdvantages from 'projects/buses/components/BusesAdvantages/BusesAdvantages';
import PreviousSearches from './components/PreviousSearches/PreviousSearches';
import IndexSubscriptionForm from '../../components/IndexSubscriptionForm/IndexSubscriptionForm';
import Text from 'components/Text/Text';
import CrossLinksGallery from 'projects/index/IndexApp/components/CrossLinksGallery/CrossLinksGallery';
import HowToBuyBusesTicket from 'projects/index/IndexApp/components/IndexBuses/components/HowToBuyBusesTicket/HowToBuyBusesTicket';

import cx from './IndexBuses.scss';

const ROOT_QA = 'indexBusesPage';

const IndexBuses: React.FC = () => {
    const nonce = useSelector(nonceSelector);
    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const location = useLocation();
    const needToRender = useServerDataFetcher([prefetchData]);

    const {fromSlug, toSlug, focus} = getIndexQuery(location);

    const crossLinksState = useSelector(crossLinksSelector);
    const {
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = crossLinksState;
    const galleryItems = useMemo(() => {
        return prepareCrossLinksForGallery(crossLinksState, EProjectName.BUSES);
    }, [crossLinksState]);

    const {rebrandingSubscriptions} = useExperiments();

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (fromSlug || toSlug) {
            reachGoal(EBusesGoal.SEARCH_OUTDATED);
        }
    }, [fromSlug, toSlug]);

    useEffect(() => {
        if (project !== EProjectName.BUSES || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.BUSES));
        }
    }, [dispatch, isRecipesFetched, project]);

    const crossLinkGalleryTitle = useMemo(() => {
        return insertJSXIntoKey(i18nBusesIndexBlock.titleNew)({
            start: (
                <Text weight="fat" size="inherit">
                    {i18nBusesIndexBlock.titleNewStart()}
                </Text>
            ),
        });
    }, []);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.BUSES;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), {
                root_rebrandingSubscriptions: rebrandingSubscriptions,
            })}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <SearchForm
                className={cx('searchForm')}
                needToSetFromSuggestByGeoPosition
                initiallyCalendarIsOpen={Boolean(fromSlug || toSlug)}
                autoFocusFieldName={focus as ESearchFormFieldName}
            />

            {deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'previousSearches',
                    })}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            <BusesAdvantages
                className={cx('advantages')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'advantages',
                })}
            />

            <IndexSubscriptionForm
                className={cx('subscribe')}
                vertical={ESubscriptionVerticalName.Bus}
            />

            {renderCrossaleBlock && (
                <CrossLinksGallery
                    className={cx('recipes')}
                    title={crossLinkGalleryTitle}
                    items={galleryItems}
                    loading={isRecipesLoading}
                    nonce={nonce}
                    markupText={i18nBusesIndexBlock.title()}
                    itemClickMetrikaGoal={ECommonGoal.RECIPE_LINK}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'crossLinksGallery',
                    })}
                />
            )}

            <HowToBuyBusesTicket
                className={cx('howToBuyATicket')}
                {...prepareQaAttributes({
                    parent: ROOT_QA,
                    current: 'howToBuyATicket',
                })}
            />
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.BUSES_SEARCH_FORM, searchFormReducer],
    [ELoadableReducer.BUSES_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.BUSES_CALENDAR, calendarReducer],
    [ELoadableReducer.BUSES_PREVIOUS_SEARCHES, previousSearchesReducer],
    [ELoadableReducer.BUSES_CONTEXT, contextReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
])(withSaga([requestPreviousSearchesSaga])(IndexBuses));
