import React, {useCallback, useMemo} from 'react';
import moment from 'moment-timezone';

import {IWithClassName} from 'types/withClassName';
import {IBusesPreviousSearch} from 'types/buses/common/IBusesPreviousSearch';

import {CurrencyType} from 'utilities/currency/CurrencyType';
import {HUMAN, HUMAN_WITH_YEAR, TIME} from 'utilities/dateUtils/formats';
import getWhenMoment from 'utilities/dateUtils/when/getWhenMoment';
import {getNow} from 'utilities/dateUtils';

import * as i18nCommonBlock from 'i18n/common';

import Text from 'components/Text/Text';
import TextSkeleton from 'components/Skeletons/TextSkeleton/TextSkeleton';
import Price from 'components/Price/Price';
import SearchPreset from 'components/SearchPreset/SearchPreset';
import Flex from 'components/Flex/Flex';

import cx from './PreviousSearch.scss';

interface IPreviousSearchProps extends IWithClassName {
    item: IBusesPreviousSearch;
    isLoading: boolean;
    index: number;
    onClick(item: IBusesPreviousSearch): void;
}

const PreviousSearch: React.FC<IPreviousSearchProps> = props => {
    const {
        className,
        item,
        item: {title, price, times, timezone, when, isToday},
        isLoading,
        index,
        onClick,
    } = props;
    const handleClick = useCallback(() => onClick(item), [onClick, item]);

    const priceNode = useMemo(() => {
        if (isLoading) {
            return (
                <TextSkeleton
                    className={cx('priceSkeleton')}
                    size="m"
                    withAnimation
                />
            );
        }

        if (!price) {
            return;
        }

        return <Price value={price} currency={CurrencyType.RUB} isRound />;
    }, [price, isLoading]);

    const secondRow = useMemo(() => {
        const size = 'm';

        if (isLoading) {
            return (
                <TextSkeleton
                    className={cx('secondRowSkeleton')}
                    size={size}
                    withAnimation
                />
            );
        }

        if (isToday) {
            if (times.length) {
                return (
                    <Flex inline between={2}>
                        {times.slice(0, 3).map(time => (
                            <Text key={time} size={size}>
                                {moment(time).tz(timezone).format(TIME)}
                            </Text>
                        ))}

                        {times.length > 3 && <Text size={size}>...</Text>}
                    </Flex>
                );
            }

            return <Text size={size}>{i18nCommonBlock.todayNoSegments()}</Text>;
        }

        if (when) {
            const whenMoment = getWhenMoment(when);
            const isAnotherYear = whenMoment.isAfter(getNow(), 'year');
            const format = isAnotherYear ? HUMAN_WITH_YEAR : HUMAN;

            return whenMoment.format(format);
        }
    }, [isLoading, isToday, times, timezone, when]);

    return (
        <SearchPreset
            className={className}
            onClick={handleClick}
            title={title}
            theme="blue"
            secondRow={secondRow}
            sideTitle={priceNode}
            index={index}
        />
    );
};

export default React.memo(PreviousSearch);
