import React, {useEffect, useMemo, useRef} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import {useLocation} from 'react-router-dom';

import {EProjectName} from 'constants/common';

import {ServerDataFetcherBag} from 'server/redux/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESearchFormFieldName} from 'components/SearchForm/types';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {IHotelsSuggest} from 'types/hotels/common/ISearchSuggest';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import searchSuggestsReducer from 'reducers/hotels/searchSuggests/reducer';
import previousSearchesReducer from 'reducers/hotels/previousSearches/reducer';
import searchInformationReducer from 'reducers/hotels/searchInformation/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';

import nonceSelector from 'selectors/common/nonceSelector';
import crossLinksSelector from 'selectors/common/crossLinksSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {loadable} from 'utilities/componentLoadable';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {reachGoal} from 'utilities/metrika';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import prepareCrossLinksForGallery from 'utilities/prepareCrossLinksForGallery';
import {getAutoFocusFieldName} from 'projects/index/IndexApp/components/IndexHotels/utilities/getAutoFocusFieldName';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-Recipes';

import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import ChunksPreloader from 'components/ChunksPreloader/ChunksPreloader';
import SearchForm from 'projects/hotels/components/SearchForm/SearchForm';
import HotelAdvantages from 'projects/hotels/components/HotelAdvantages/HotelAdvantages';
import GeolocationError from 'projects/hotels/components/GeolocationError/GeolocationError';
import PreviousSearches from './components/PreviousSearches/PreviousSearches';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';
import IndexSubscriptionForm from '../../components/IndexSubscriptionForm/IndexSubscriptionForm';
import CrossLinksGallery from 'projects/index/IndexApp/components/CrossLinksGallery/CrossLinksGallery';
import Text from 'components/Text/Text';

import HotelsSearchInformationProvider from 'projects/hotels/containers/HotelsSearchInformationProvider/HotelsSearchInformationProvider';

import prefetchCrossLinks from 'server/redux/common/fetchCrossLinks';

import cx from './IndexHotels.scss';

async function fetchIndexHotelsInfoSSR(
    props: ServerDataFetcherBag,
): Promise<void> {
    await prefetchCrossLinks(props, EProjectName.HOTELS);
}

const PRELOAD_CHUNKS = [
    loadable(
        () => import('projects/hotels/pages/SearchPage/SearchPageContainer'),
    ),
];

const ROOT_QA = 'indexHotelsPage';

const IndexHotels: React.FC = () => {
    const crossLinksState = useSelector(crossLinksSelector);
    const {
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = crossLinksState;
    const galleryItems = useMemo(() => {
        return prepareCrossLinksForGallery(
            crossLinksState,
            EProjectName.HOTELS,
        );
    }, [crossLinksState]);

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const location = useLocation();
    const nonce = useSelector(nonceSelector);

    const {rebrandingSubscriptions} = useExperiments();

    const needToRender = useServerDataFetcher([fetchIndexHotelsInfoSSR]);

    const {hotelsPreviousSearches: hotelsPreviousSearchesExp} =
        useExperiments();

    const searchFormFieldsRef = useRef<IFormFieldsRef>();

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    const onChangeToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IHotelsSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.HOTELS_MORDA_SEARCH_FORM_CLEAR_TO);
            }
        },
    );

    useEffect(() => {
        if (project !== EProjectName.HOTELS || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.HOTELS));
        }
    }, [dispatch, isRecipesFetched, project]);

    const crossLinkGalleryTitle = useMemo(() => {
        return insertJSXIntoKey(i18nBlock.title)({
            start: (
                <Text weight="fat" size="inherit">
                    {i18nBlock.searchHotels()}
                </Text>
            ),
            end: (
                <Text weight="fat" size="inherit">
                    {i18nBlock.atCompetitivePrices()}
                </Text>
            ),
        });
    }, []);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.HOTELS;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), {
                root_rebrandingSubscriptions: rebrandingSubscriptions,
            })}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <HotelsSearchInformationProvider />
            <ChunksPreloader
                preloadChunks={PRELOAD_CHUNKS}
                maxProportionTimeout={15000}
            />

            <SearchForm
                className={cx('searchForm')}
                autoFocusFieldName={
                    getAutoFocusFieldName(location) as ESearchFormFieldName
                }
                fieldsRef={searchFormFieldsRef}
                onChangeToPoint={onChangeToPoint}
            />

            {hotelsPreviousSearchesExp && deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    searchFormFieldsRef={searchFormFieldsRef}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            <HotelAdvantages className={cx('advantages')} />

            <IndexSubscriptionForm
                className={cx('subscribe')}
                vertical={ESubscriptionVerticalName.Hotels}
            />

            {renderCrossaleBlock && (
                <CrossLinksGallery
                    className={cx('recipes')}
                    title={crossLinkGalleryTitle}
                    items={galleryItems}
                    loading={isRecipesLoading}
                    nonce={nonce}
                    markupText={i18nBlock.blockDashTitleDashNew()}
                    itemClickMetrikaGoal={ECommonGoal.RECIPE_LINK}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'crossLinksGallery',
                    })}
                />
            )}

            <GeolocationError />
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.HOTELS_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.HOTELS_SEARCH_INFORMATION, searchInformationReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
    [ELoadableReducer.HOTELS_PREVIOUS_SEARCHES, previousSearchesReducer],
])(IndexHotels);
