import React, {MutableRefObject, useCallback, useMemo} from 'react';
import {useHistory} from 'react-router-dom';
import {useDispatch} from 'react-redux';

import {ESearchFormFieldName} from 'components/SearchForm/types';
import {IHotelsPreviousSearch} from 'types/hotels/previousSearches/IPreviousSearch';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import fillSearchFormByPreviousSearchThunkAction from 'reducers/hotels/previousSearches/thunk/fillSearchFormByPreviousSearch';

import {hotelsURLs} from 'projects/hotels/utilities/urls';
import {reachGoal} from 'utilities/metrika';
import getPreviousSearchDateDescription from 'projects/hotels/utilities/previousSearches/getPreviousSearchDateDescription';

import * as i18n from 'i18n/common';

import SearchPreset from 'components/SearchPreset/SearchPreset';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import {ESuggestSource} from 'components/SearchSuggest/SearchSuggest';

interface IPreviousSearchProps {
    item: IHotelsPreviousSearch;
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined>;
    index: number;
}

const PreviousSearch: React.FC<IPreviousSearchProps> = props => {
    const {item, searchFormFieldsRef, index} = props;
    const history = useHistory();
    const dispatch = useDispatch();

    const handleClick = useCallback(() => {
        dispatch(fillSearchFormByPreviousSearchThunkAction(item));

        if (item.startDate) {
            history?.push(
                hotelsURLs.getSearchUrl(
                    {
                        searchFormFieldsValues: {
                            [ESearchFormFieldName.TO]: {
                                selectedValue: item.to,
                                inputValue: item.to.name,
                                source: ESuggestSource.PRESETS,
                            },
                            [ESearchFormFieldName.START_DATE]: item.startDate,
                            [ESearchFormFieldName.END_DATE]: item.endDate,
                            adults: item.adults,
                            childrenAges: item.childrenAges,
                        },
                    },
                    {
                        hasProductionOffers: false,
                        hasDirectionSuggest: true,
                    },
                ),
            );
        } else {
            searchFormFieldsRef.current?.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        }

        reachGoal(EHotelsGoal.PREVIOUS_SEARCHES_PRESET_CLICK);
    }, [dispatch, item, searchFormFieldsRef, history]);

    const secondRow = useMemo(() => {
        return getPreviousSearchDateDescription(item) || i18n.selectDates();
    }, [item]);

    return (
        <SearchPreset
            onClick={handleClick}
            title={item.to.name}
            theme="blue"
            secondRow={secondRow}
            index={index}
        />
    );
};

export default React.memo(PreviousSearch);
