import {ComponentType, FC, RefObject} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IIconProps} from 'icons/types/icon';

import {internalUrl} from 'utilities/url';
import {
    IWithQaAttributes,
    prepareCustomQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';

import Flex from 'components/Flex/Flex';
import Link from 'components/Link/Link';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';

import cx from './TabLink.scss';

interface ITabLinkProps extends IWithClassName, IWithQaAttributes {
    isActive: boolean;
    tabLink: string;
    Icon?: ComponentType<IIconProps>;
    title: string;
    innerRef?: RefObject<HTMLAnchorElement>;
    onClick(): void;
}

const TabLink: FC<ITabLinkProps> = props => {
    const {className, isActive, tabLink, Icon, title, onClick, innerRef} =
        props;

    const deviceType = useDeviceType();

    return (
        <Link
            className={cx(
                className,
                'root',
                {root_active: isActive},
                deviceMods('root', deviceType),
            )}
            to={internalUrl(tabLink)}
            theme="black"
            innerRef={innerRef}
            onClick={onClick}
            {...prepareQaAttributes(props)}
            {...prepareCustomQaAttributes('data-active', String(isActive))}
        >
            {deviceType.isMobile ? (
                <Flex flexDirection="column" alignItems="center">
                    {Icon && (
                        <Flex
                            alignItems="center"
                            justifyContent="center"
                            className={cx('iconWrapper')}
                        >
                            <Icon />
                        </Flex>
                    )}
                    <span className={cx('title')}>{title.toLowerCase()}</span>
                </Flex>
            ) : (
                <TextWithIcon
                    className={cx('textWithIcon')}
                    textClassName={cx('title')}
                    text={title}
                    iconLeft={Icon}
                    size="l"
                    spaceBetween="3"
                />
            )}
        </Link>
    );
};

export default TabLink;
