import {useDispatch, useSelector} from 'react-redux';
import {useEffect} from 'react';

import {ServerDataFetcherBag} from 'server/redux/types';

import {fetchWidgetInfo} from 'reducers/tours/widgetInfo/thunk';
import {fillSearchSuggestFromByGeobase} from 'reducers/tours/searchSuggests/thunk';

import nonceSelector from 'selectors/common/nonceSelector';
import widgetInfoSelector from 'selectors/tours/widgetInfo/widgetInfoSelector';
import searchFormSelector from 'selectors/tours/searchForm/searchFormSelector';

import {deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';

import SearchForm from './components/SearchForm/SearchForm';
import ToursAdvantages from './components/ToursAdvantages/ToursAdvantages';
import PlusInformation from './components/PlusInformation/PlusInformation';
import LevelTravelBanner from './components/LevelTravelBanner/LevelTravelBanner';

import {prefillToursSearchForm} from 'server/redux/tours/prefillSearchForm';

import cx from './IndexTours.scss';

async function prefetchData(bag: ServerDataFetcherBag): Promise<void> {
    await Promise.all([prefillToursSearchForm(bag)]);
}

const IndexTours: React.FC = () => {
    const deviceType = useDeviceType();

    const nonce = useSelector(nonceSelector);
    const {isError, isLoading, label} = useSelector(widgetInfoSelector);
    const {fromField} = useSelector(searchFormSelector);
    const dispatch = useDispatch();

    const needToRender = useServerDataFetcher([prefetchData]);

    useEffect(() => {
        if (!fromField.selectedValue)
            dispatch(fillSearchSuggestFromByGeobase({}));
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [dispatch]);

    useEffect(() => {
        if (!(isError || isLoading) && !label) {
            dispatch(fetchWidgetInfo());
        }
    }, [dispatch, isError, isLoading, label]);

    if (!needToRender) {
        return null;
    }

    return (
        <div className={cx('root', deviceMods('root', deviceType))}>
            <PlusInformation
                className={cx('plusInformation')}
                deviceType={deviceType}
            />
            <SearchForm
                label={label}
                nonce={nonce}
                deviceType={deviceType}
                nightsClassName={cx('nights')}
            />
            <LevelTravelBanner className={cx('banner')} />
            <ToursAdvantages className={cx('advantages')} />
        </div>
    );
};

export default IndexTours;
