import React, {useCallback, useEffect, useMemo} from 'react';
import {useDispatch, useSelector} from 'react-redux';
import isEmpty from 'lodash/isEmpty';

import {TOURS_SEARCH_FROM_NAME} from 'constants/reduxForm';

import {IWithDeviceType} from 'types/withDeviceType';
import {
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {
    ESuggestDestinationType,
    IToursSearchFormErrors,
    IToursSearchFormFromPointField,
    IToursSearchFormToPointField,
    IToursSuggestDeparture,
    IToursSuggestDestination,
} from './types';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {EToursGoal} from 'utilities/metrika/types/goals/tours';

import requestSearchSuggestsThunkAction from 'reducers/tours/searchSuggests/thunk';
import {
    setToursSearchFormAdultsFieldAction,
    setToursSearchFormChildrenAgesFieldAction,
    setToursSearchFormFlexibleDatesAction,
    setToursSearchFormFlexibleNightsAction,
    setToursSearchFormFromFieldAction,
    setToursSearchFormNightsFieldAction,
    setToursSearchFormStartDateFieldAction,
    setToursSearchFormToFieldAction,
} from 'reducers/tours/searchForm/actions';

import searchFormSelector from 'selectors/tours/searchForm/searchFormSelector';
import {getUserInfo} from 'selectors/common/userInfoSelector';

import {deviceModMobile} from 'utilities/stylesUtils';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getSuggestTitleAndDescription from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/utilities/getSuggestTitleAndDescription';
import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import validateForm from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/utilities/validateForm';
import {reachGoal} from 'utilities/metrika';
import getRedirect from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/utilities/getRedirect';
import {useIsAuth} from 'utilities/hooks/useIsAuth';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import {getSearchStateQuery} from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/utilities/getSearchStateQuery';

import * as i18nBlock from 'i18n/tours-SearchForm';
import * as i18nBlockTours from 'i18n/tours';

import CommonSearchForm, {
    IBaseSearchFormProps,
} from 'components/SearchForm/SearchForm';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import useSuggestInputFocusHandler from 'components/SearchForm/hooks/useSuggestInputFocusHandler';
import ToursTravellers from './components/ToursTravellers/ToursTravellers';
import ToursNightsCount from './components/ToursNightsCount/ToursNightsCount';
import ToursUnauthorizedSubmitDialog from './components/UnauthorizedSubmitDialog/ToursUnauthorizedSubmitDialog';
import DateAccuracyToggle from 'projects/index/IndexApp/components/IndexTours/components/SearchForm/components/DateAccuracyToggle/DateAccuracyToggle';
import {ESearchControlPlaceholderPosition} from 'components/SearchControl/SearchControl';

import cx from './SearchForm.scss';

const FIELD_NAMES = [
    ESearchFormFieldName.FROM,
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
];

interface ISearchFormProps extends IWithDeviceType, IBaseSearchFormProps {
    nonce: string;
    label: string | null;
    nightsClassName?: string;
    travellersClassName?: string;
    travellersTriggerClassName?: string;
    datePickerTriggerClassName?: string;
    datePickerPlaceholderPosition?: ESearchControlPlaceholderPosition;
    onChangeFromPoint?(value: ISuggestValue<IToursSuggestDeparture>): void;
    onChangeToPoint?(value: ISuggestValue<IToursSuggestDestination>): void;
}

const SearchForm: React.FC<ISearchFormProps> = ({
    label,
    deviceType,
    onChangeFromPoint,
    size = ESearchFormSize.XL,
    triggerViewType = ESearchFormTriggerViewType.UNION,
    canToggleDropdowns = true,
    canSubmitOnChange = false,
    onInteract,
    onShowFieldPopup,
    onHideFieldPopup,
    isExpanded,
    onChangeToPoint,
    travellersClassName,
    nightsClassName,
    travellersTriggerClassName,
    fieldsRef,
}) => {
    const {
        fromField,
        toField,
        startDateField,
        flexibleDatesField,
        flexibleNightsField,
        nightsField,
        adultsField,
        childrenAgesField,
        searchSuggests,
    } = useSelector(searchFormSelector);
    const isAuth = useIsAuth();
    const dispatch = useDispatch();
    const {passportPath} = useSelector(getUserInfo);
    const unauthorizedDialogOpen = useBoolean(false);

    const requestSuggests = useImmutableCallback(
        (
            fromField: ISuggestValue<IToursSuggestDeparture>,
            toField: ISuggestValue<IToursSuggestDestination>,
        ) => {
            dispatch(
                requestSearchSuggestsThunkAction({
                    fromField,
                    toField,
                }),
            );
        },
    );

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const handleSuggestInputFocus = useSuggestInputFocusHandler<
        IToursSuggestDeparture,
        IToursSuggestDestination
    >(fromField, toField, requestSuggestsWithThrottle);

    useEffect(() => {
        requestSuggestsWithThrottle(fromField, toField);
    }, [fromField, requestSuggestsWithThrottle, toField]);

    const setFromPoint = useImmutableCallback(
        (fieldValue: IToursSearchFormFromPointField) => {
            dispatch(setToursSearchFormFromFieldAction(fieldValue));

            onChangeFromPoint?.(fieldValue);
        },
    );

    const setToPoint = useImmutableCallback(
        (fieldValue: IToursSearchFormToPointField) => {
            dispatch(setToursSearchFormToFieldAction(fieldValue));

            onChangeToPoint?.(fieldValue);
        },
    );

    const setStartDate = useDispatchedAction(
        setToursSearchFormStartDateFieldAction,
        dispatch,
    );

    const setFlexibleNights = useDispatchedAction(
        setToursSearchFormFlexibleNightsAction,
        dispatch,
    );

    const handleFlexibleNightsToggle = useCallback(
        () => setFlexibleNights(!flexibleNightsField),
        [flexibleNightsField, setFlexibleNights],
    );

    const setFlexibleDates = useDispatchedAction(
        setToursSearchFormFlexibleDatesAction,
        dispatch,
    );

    const handleFlexibleDatessToggle = useCallback(
        () => setFlexibleDates(!flexibleDatesField),
        [flexibleDatesField, setFlexibleDates],
    );

    const setNights = useDispatchedAction(
        setToursSearchFormNightsFieldAction,
        dispatch,
    );

    const setAdults = useDispatchedAction(
        setToursSearchFormAdultsFieldAction,
        dispatch,
    );

    const setChildrenAges = useDispatchedAction(
        setToursSearchFormChildrenAgesFieldAction,
        dispatch,
    );

    const handleTravelersFocus = useImmutableCallback(() => {
        // Должно срабатывать после скрытия других попапов
        requestAnimationFrame(() => {
            onInteract?.(true);
        });
    });

    const getTravelersBlurHandler = useImmutableCallback(
        (submit: () => any) => {
            return (): void => {
                onInteract?.(false);

                if (canSubmitOnChange) {
                    requestAnimationFrame(() => {
                        submit();
                    });
                }
            };
        },
    );

    const redirectAfterSubmit = useImmutableCallback(() => {
        const from = fromField.selectedValue as IToursSuggestDeparture;
        const to = toField.selectedValue as IToursSuggestDestination;

        getRedirect({
            label,
            hotel_ids:
                to.type === ESuggestDestinationType.HOTEL
                    ? to.hotel_ids
                    : undefined,
            to_country: to.iso2,
            to_city:
                to.type !== ESuggestDestinationType.COUNTRY
                    ? to.name_en
                    : undefined,
            from_city: from.name_en,
            from_country: from.iso2,
            nights: nightsField,
            start_date: startDateField,
            adults: adultsField,
            kids: childrenAgesField ? childrenAgesField.length : undefined,
            kids_ages: childrenAgesField,
            flex_nights: flexibleNightsField,
            flex_dates: flexibleDatesField,
        });
    });

    const handleSubmit = useImmutableCallback(
        ({
            handleTrySubmit,
            formErrors,
            setHasSubmitFailed,
        }: {
            handleTrySubmit(): void;
            formErrors: IToursSearchFormErrors;
            setHasSubmitFailed(val: boolean): void;
        }) => {
            handleTrySubmit();

            if (!isEmpty(formErrors)) {
                return setHasSubmitFailed(true);
            }

            reachGoal(EIndexGoal.TOURS_SEARCH_BUTTON);

            if (isAuth) {
                redirectAfterSubmit();
            } else {
                reachGoal(EToursGoal.TOURS_MODAL_FORCE_LOGIN_SHOW);
                unauthorizedDialogOpen.setTrue();
            }
        },
    );

    const handleCloseUnauthorized = useImmutableCallback(() => {
        unauthorizedDialogOpen.setFalse();
        reachGoal(EToursGoal.TOURS_MODAL_FORCE_LOGIN_CLOSE);
    });
    const handleSkipAuthorization = useImmutableCallback(() => {
        unauthorizedDialogOpen.setFalse();
        redirectAfterSubmit();
    });
    const handleAuthorization = useImmutableCallback(() => {
        const retpath = new URL(location.href);
        const query = getSearchStateQuery({
            to: toField,
            from: fromField,
            nightsCount: nightsField,
            adults: adultsField,
            childrenAges: childrenAgesField,
            startDate: startDateField,
            flexibleDates: flexibleDatesField,
            flexibleNights: flexibleNightsField,
        });

        if (!query) return;

        Object.entries(query).forEach(([key, value]) =>
            retpath.searchParams.set(key, value.toString()),
        );

        const url = getPassportUrl({
            mode: EPassportMode.AUTH,
            passportHost: passportPath,
            retpath: retpath.toString(),
        });

        window.open(url, '_self');
        reachGoal(EToursGoal.TOURS_MODAL_FORCE_LOGIN_LOGIN_BUTTON_CLICK);
    });

    const datePickerHeaderBlock = useMemo(() => {
        return (
            <DateAccuracyToggle
                className={cx('dateAccuracyBlock')}
                toggleClassName={cx(
                    deviceType.isDesktop && 'dateAccuracyToggle',
                )}
                text={i18nBlock.flexibleDateText()}
                isActive={flexibleDatesField}
                justifyContent={deviceType.isDesktop ? 'flex-start' : undefined}
                onToggle={handleFlexibleDatessToggle}
            />
        );
    }, [deviceType.isDesktop, flexibleDatesField]);

    const renderTravellersBlock = useCallback(
        ({baseCx, submit}: ISearchFormRenderTravelersBlockProps) => {
            return (
                <>
                    <div className={baseCx('travellers', nightsClassName)}>
                        <ToursNightsCount
                            triggerClassName={baseCx(
                                'travellersTrigger',
                                travellersTriggerClassName,
                            )}
                            triggerFocusClassName={baseCx(
                                'travellersTrigger_focus',
                            )}
                            size={size}
                            triggerViewType={triggerViewType}
                            tabIndex={isExpanded ? 0 : -1}
                            nightsCount={nightsField}
                            setNights={setNights}
                            isFlexibleNights={flexibleNightsField}
                            handleFlexibleNightsToggle={
                                handleFlexibleNightsToggle
                            }
                            deviceType={deviceType}
                            canToggle={canToggleDropdowns}
                            onShowPopup={onShowFieldPopup}
                            onHidePopup={onHideFieldPopup}
                            onFocus={handleTravelersFocus}
                            onBlur={getTravelersBlurHandler(submit)}
                        />
                    </div>
                    <div className={baseCx('travellers', travellersClassName)}>
                        <ToursTravellers
                            triggerClassName={baseCx(
                                'travellersTrigger',
                                'lastControl',
                                travellersTriggerClassName,
                            )}
                            triggerFocusClassName={baseCx(
                                'travellersTrigger_focus',
                            )}
                            size={size}
                            triggerViewType={triggerViewType}
                            tabIndex={isExpanded ? 0 : -1}
                            adultsCount={adultsField}
                            setAdults={setAdults}
                            childrenAges={childrenAgesField}
                            setChildrenAges={setChildrenAges}
                            deviceType={deviceType}
                            canToggle={canToggleDropdowns}
                            onShowPopup={onShowFieldPopup}
                            onHidePopup={onHideFieldPopup}
                            onFocus={handleTravelersFocus}
                            onBlur={getTravelersBlurHandler(submit)}
                        />
                    </div>
                </>
            );
        },
        [
            nightsClassName,
            travellersTriggerClassName,
            size,
            triggerViewType,
            isExpanded,
            nightsField,
            setNights,
            flexibleNightsField,
            handleFlexibleNightsToggle,
            deviceType,
            canToggleDropdowns,
            onShowFieldPopup,
            onHideFieldPopup,
            handleTravelersFocus,
            getTravelersBlurHandler,
            travellersClassName,
            adultsField,
            setAdults,
            childrenAgesField,
            setChildrenAges,
        ],
    );

    return (
        <div className={cx(deviceType.isMobile && 'root_mobile')}>
            <ToursUnauthorizedSubmitDialog
                open={unauthorizedDialogOpen.value}
                onClose={handleCloseUnauthorized}
                onAuthorization={handleAuthorization}
                onSkipAuthorization={handleSkipAuthorization}
            />
            <CommonSearchForm
                className={cx(deviceModMobile('root', deviceType))}
                searchFormName={TOURS_SEARCH_FROM_NAME}
                fieldsNames={FIELD_NAMES}
                fromField={fromField}
                toField={toField}
                withoutReverseDirection
                startDateField={startDateField}
                fromSearchSuggests={
                    searchSuggests[ESearchFormFieldName.FROM].value as
                        | IToursSuggestDeparture[]
                        | null
                }
                datePickerClassName={cx(
                    deviceType.isMobile && 'travellersTrigger_mobile',
                )}
                datePickerPlaceholderPosition={
                    ESearchControlPlaceholderPosition.CENTER
                }
                datePickerHeaderBlock={datePickerHeaderBlock}
                startDateTrailingValue={
                    flexibleDatesField ? i18nBlockTours.flexValue() : undefined
                }
                toSearchSuggests={
                    searchSuggests[ESearchFormFieldName.TO].value as
                        | IToursSuggestDestination[]
                        | null
                }
                validateForm={validateForm}
                uniqueSuggestValueName="name_ru"
                getSuggestTitleAndDescription={getSuggestTitleAndDescription}
                fromPointPlaceholder={i18nBlock.fromPlaceholder()}
                toPointPlaceholder={i18nBlock.toPlaceholder()}
                startDatePlaceholder={i18nBlock.startDatePlaceholder()}
                endDatePlaceholder={i18nBlock.endDatePlaceholder()}
                setFromPoint={setFromPoint}
                setToPoint={setToPoint}
                setStartDate={setStartDate}
                onSubmit={handleSubmit}
                fieldsRef={fieldsRef}
                renderTravellersBlock={renderTravellersBlock}
                onSuggestInputFocus={handleSuggestInputFocus}
            />
        </div>
    );
};

export default SearchForm;
