import React, {PureComponent} from 'react';

import {MIN_NIGHT_COUNT, MAX_NIGHTS_COUNT} from 'constants/tours';

import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCommonPluralBlock from 'i18n/common-plural';
import * as i18nBlock from 'i18n/tours';
import * as i18nSearchFormBlock from 'i18n/tours-SearchForm';

import Travellers from 'components/Travellers/Travellers';

import DateAccuracyToggle from '../DateAccuracyToggle/DateAccuracyToggle';

import cx from '../ToursTravellers/ToursTravellers.scss';
import cy from './ToursNightsCount.scss';

const {TravellersCount} = Travellers;

interface IToursTravellersProps extends IWithQaAttributes {
    triggerClassName?: string;
    triggerFocusClassName?: string;
    size: ESearchFormSize;
    triggerViewType: ESearchFormTriggerViewType;
    nightsCount: number;
    isFlexibleNights: boolean;
    handleFlexibleNightsToggle: () => void;
    setNights(count: number): void;
    tabIndex?: number;
    canToggle: boolean;
    deviceType: IDevice;
    onFocus?(): void;
    onBlur(): void;
    onShowPopup?(): void;
    onHidePopup?(): void;
}

class ToursNightsCount extends PureComponent<IToursTravellersProps> {
    handleChangeNightsCount = (value: number): void => {
        const {setNights} = this.props;

        setNights(value);
    };

    renderTravellersButtonContent = (): React.ReactNode => {
        const {nightsCount, isFlexibleNights} = this.props;

        if (!nightsCount) {
            return (
                <div className={cx('travellersButtonPlaceholder')}>
                    {i18nBlock.nightsCount()}
                </div>
            );
        }

        const nightsValue = i18nCommonPluralBlock.nights({count: nightsCount});

        return (
            <div className={cx('travellersButtonValue')}>
                {nightsValue}
                {isFlexibleNights && i18nBlock.flexValue()}
            </div>
        );
    };

    renderNightsCount = (): React.ReactNode => {
        const {nightsCount} = this.props;

        return (
            <TravellersCount
                value={nightsCount}
                min={MIN_NIGHT_COUNT}
                max={MAX_NIGHTS_COUNT}
                onChange={this.handleChangeNightsCount}
                label={i18nBlock.nightsLabel()}
            />
        );
    };

    render(): React.ReactNode {
        const {
            deviceType,
            triggerViewType,
            triggerClassName,
            triggerFocusClassName,
            onFocus,
            onBlur,
            tabIndex,
            onShowPopup,
            onHidePopup,
            size,
            canToggle,
            isFlexibleNights,
            handleFlexibleNightsToggle,
        } = this.props;
        const {isMobile} = deviceType;

        return (
            <Travellers
                triggerClassName={triggerClassName}
                triggerFocusClassName={triggerFocusClassName}
                deviceType={deviceType}
                isModalView={isMobile}
                triggerViewType={triggerViewType}
                type="travellers"
                buttonComponent={this.renderTravellersButtonContent}
                onShowPopup={onShowPopup}
                onHidePopup={onHidePopup}
                onFocus={onFocus}
                onBlur={onBlur}
                tabIndex={tabIndex}
                size={size}
                canToggle={canToggle}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'travellers',
                })}
            >
                {this.renderNightsCount()}
                <DateAccuracyToggle
                    isActive={isFlexibleNights}
                    onToggle={handleFlexibleNightsToggle}
                    text={i18nSearchFormBlock.flexibleNightsText()}
                    className={cy('dateAccuracyToggle')}
                />
            </Travellers>
        );
    }
}

export default ToursNightsCount;
