import React, {PureComponent} from 'react';
import _times from 'lodash/times';

import {
    MIN_ADULTS_COUNT,
    MAX_ADULTS_COUNT,
    MAX_CHILDREN_AGE,
    MAX_CHILDREN_COUNT,
} from 'constants/tours';

import {
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {ECountFieldType} from 'types/common/ECountFieldType';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nCommonPluralBlock from 'i18n/common-plural';
import * as i18nBlock from 'i18n/hotels'; // toDO: В идеале развязать это, ключи не должны быть связаны с отельными кейсетами, МИНОР

import Travellers from 'components/Travellers/Travellers';

import cx from './ToursTravellers.scss';

const {TravellersCount, TravellersSelect} = Travellers;

const allChildrenItems = _times(MAX_CHILDREN_AGE + 1, age => ({
    value: age,
    text: i18nBlock.componentsDotHotelsTravellersDotChildrenAge({age}),
}));

// For native select only
const disabledFirstElementValue = -1;
const disabledFirstElement = {
    value: disabledFirstElementValue,
    text: i18nBlock.componentsDotHotelsTravellersDotChildrenAgeTitle(),
    disabled: true,
};

allChildrenItems.unshift(disabledFirstElement);

interface IToursTravellersProps extends IWithQaAttributes {
    triggerClassName?: string;
    triggerFocusClassName?: string;
    size: ESearchFormSize;
    triggerViewType: ESearchFormTriggerViewType;
    adultsCount: number;
    childrenAges: number[];
    setAdults(count: number): void;
    setChildrenAges(ages: number[]): void;
    tabIndex?: number;
    canToggle: boolean;
    deviceType: IDevice;
    onFocus?(): void;
    onBlur(): void;
    onShowPopup?(): void;
    onHidePopup?(): void;
}

class ToursTravellers extends PureComponent<IToursTravellersProps> {
    handleChangeTravellersCount = (value: number): void => {
        const {setAdults} = this.props;

        setAdults(value);
    };

    handleChangeTravellersSelect = ({values}: {values: number[]}): void => {
        const {setChildrenAges} = this.props;

        setChildrenAges(values);
    };

    renderSelectedItem = (age: number): string => {
        return i18nBlock.componentsDotHotelsTravellersDotChildrenTitle({age});
    };

    renderTravellersButtonContent = (): React.ReactNode => {
        const {adultsCount, childrenAges} = this.props;

        const childrenCount = childrenAges.length;
        const totalTravellers = Number(adultsCount + childrenCount);

        if (!totalTravellers) {
            return (
                <div className={cx('travellersButtonPlaceholder')}>
                    {i18nBlock.guestsCount()}
                </div>
            );
        }

        const travellersValue = childrenCount
            ? i18nCommonPluralBlock.guests({count: totalTravellers})
            : i18nBlock.componentsDotHotelsTravellersDotAdultsCount({
                  count: totalTravellers,
              });

        return (
            <div className={cx('travellersButtonValue')}>{travellersValue}</div>
        );
    };

    renderTravellersCount = (): React.ReactNode => {
        const {adultsCount} = this.props;

        return (
            <TravellersCount
                value={adultsCount}
                min={MIN_ADULTS_COUNT}
                max={MAX_ADULTS_COUNT}
                onChange={this.handleChangeTravellersCount}
                label={i18nBlock.componentsDotHotelsTravellersDotAdultsTitle()}
                hint={i18nBlock.componentsDotHotelsTravellersDotAdultsAgeTitle()}
                {...prepareQaAttributes({
                    parent: 'travellersCount',
                    key: ECountFieldType.ADULTS,
                })}
            />
        );
    };

    renderTravellersSelect = (): React.ReactNode => {
        const {childrenAges} = this.props;

        return (
            <TravellersSelect
                values={childrenAges}
                type="children"
                items={allChildrenItems}
                buttonLabel={i18nBlock.componentsDotHotelsTravellersDotAddChildren()}
                className={cx('travellersSelect')}
                maxValuesCount={MAX_CHILDREN_COUNT}
                onChange={this.handleChangeTravellersSelect}
                renderSelectedItem={this.renderSelectedItem}
                nativeSelectProps={{value: disabledFirstElementValue}}
            />
        );
    };

    render(): React.ReactNode {
        const {
            deviceType,
            triggerViewType,
            triggerClassName,
            triggerFocusClassName,
            onFocus,
            onBlur,
            tabIndex,
            onShowPopup,
            onHidePopup,
            size,
            canToggle,
        } = this.props;
        const {isMobile} = deviceType;

        return (
            <Travellers
                triggerClassName={triggerClassName}
                triggerFocusClassName={triggerFocusClassName}
                deviceType={deviceType}
                isModalView={isMobile}
                triggerViewType={triggerViewType}
                type="travellers"
                buttonComponent={this.renderTravellersButtonContent}
                onShowPopup={onShowPopup}
                onHidePopup={onHidePopup}
                onFocus={onFocus}
                onBlur={onBlur}
                tabIndex={tabIndex}
                size={size}
                canToggle={canToggle}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'travellers',
                })}
            >
                {this.renderTravellersCount()}
                {this.renderTravellersSelect()}
            </Travellers>
        );
    }
}

export default ToursTravellers;
