import {FC, memo} from 'react';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModDesktop, deviceModMobile} from 'utilities/stylesUtils';

import * as i18n from 'i18n/tours-ToursAdvantages';
import * as i18nCommon from 'i18n/common';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Heading from 'components/Heading/Heading';
import Button from 'components/Button/Button';
import PlusPromoM from 'icons/illustrations/PlusPromo/PlusPromoM';
import PlusPromoL from 'icons/illustrations/PlusPromo/PlusPromoL';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';

import cx from './UnauthorizedSubmitDialog.scss';

export interface IToursUnauthorizedSubmitDialogProps {
    open: boolean;
    onClose(): void;
    onAuthorization?(): void;
    onSkipAuthorization?(): void;
}

const ToursUnauthorizedSubmitDialog: FC<IToursUnauthorizedSubmitDialogProps> =
    ({
        onAuthorization,
        onSkipAuthorization,
        open,
        onClose,
    }: IToursUnauthorizedSubmitDialogProps) => {
        const device = useDeviceType();

        return (
            <ModalOrBottomSheet
                isVisible={open}
                onClose={onClose}
                modalProps={{
                    containerClassName: cx('modal'),
                }}
                bottomSheetProps={{
                    className: cx('modal'),
                }}
            >
                <div
                    className={cx(
                        'illustration',
                        deviceModMobile('illustration', device),
                    )}
                >
                    {device.isMobile ? <PlusPromoM /> : <PlusPromoL />}
                </div>
                <div
                    className={cx(
                        'content',
                        deviceModMobile('content', device),
                    )}
                >
                    <Heading level="2" className={cx('heading')}>
                        {i18n.plusUnauthorizedDialogTitle()}
                    </Heading>
                    <Text
                        tag="div"
                        size="m"
                        className={cx('body', deviceModDesktop('body', device))}
                    >
                        {i18n.plusUnauthorizedDialogContent()}
                    </Text>
                    <Flex
                        inline
                        between={3}
                        justifyContent="center"
                        alignItems="center"
                    >
                        <Button
                            size="l"
                            theme="secondary"
                            onClick={onSkipAuthorization}
                            className={cx(
                                'skipButton',
                                deviceModDesktop('skipButton', device),
                            )}
                        >
                            {i18n.plusModalSecondaryButtonText()}
                        </Button>
                        <Button
                            size="l"
                            theme="primary"
                            onClick={onAuthorization}
                            className={cx(
                                'loginButton',
                                deviceModDesktop('loginButton', device),
                            )}
                        >
                            {i18nCommon.actionsDotLogIn()}
                        </Button>
                    </Flex>
                </div>
            </ModalOrBottomSheet>
        );
    };

export default memo(ToursUnauthorizedSubmitDialog);
