import React, {useMemo} from 'react';
import {useSelector} from 'react-redux';

import {YANDEX_PLUS_URL} from 'constants/urls/plus';

import {IWithClassName} from 'types/withClassName';

import {getUserInfo} from 'selectors/common/userInfoSelector';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';
import {deviceMods} from 'utilities/stylesUtils';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/tours-ToursAdvantages';
import * as i18nRebranding from 'i18n/index-Advantages';

import Advantages, {TAdvantage} from 'components/Advantages/Advantages';
import Link from 'components/Link/Link';
import InfoPopup from 'components/InfoPopup/InfoPopup';
import Text from 'components/Text/Text';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import PlusColoredLabel from 'components/PlusColoredLabel/PlusColoredLabel';
import {NoSsr} from 'components/NoSsr';
import useTitle from 'components/Advantages/hooks/useTitle';
import PlusCashback3DImage from 'icons/raster-illustrations/M/PlusCashback3D';
import Fair3DImage from 'icons/raster-illustrations/M/Fair3D';
import Tours3DImage from 'icons/raster-illustrations/M/Tours3D';
import Support3DImage from 'icons/raster-illustrations/M/Support3D';

import cx from './ToursAdvantages.scss';

interface IToursAdvantages extends IWithClassName, IWithQaAttributes {}

const ToursAdvantages: React.FC<IToursAdvantages> = props => {
    const {className} = props;
    const userInfo = useSelector(getUserInfo);
    const deviceType = useDeviceType();

    const plus = useMemo(
        () => (
            <Link
                target="_blank"
                rel="noopener noreferrer"
                url={YANDEX_PLUS_URL}
                className={cx(deviceMods('link', deviceType))}
            >
                {isAuthUser(userInfo) ? i18nBlock.plus2() : i18nBlock.plus()}
            </Link>
        ),
        [deviceType, userInfo],
    );

    const plusTooltip = useMemo(
        () => (
            <InfoPopup
                messageBoxProps={{theme: EMessageBoxPopupTheme.WHITE}}
                className={cx('hint')}
                contentClassName={cx('hintPopup')}
            >
                <div
                    className={cx(
                        'hintContent',
                        deviceMods('hintContent', deviceType),
                    )}
                >
                    {i18nBlock.plusAdvantageHint()}
                </div>
            </InfoPopup>
        ),
        [deviceType],
    );

    const plusLimitInfo = useMemo(
        () => (
            <Text className={cx('limitInfo')} whiteSpace="nowrap" size="m">
                {insertJSXIntoKey(i18nBlock.plusAdvantageLimitInfo)({
                    plusIcon: <PlusColoredLabel size="s" count={5000} />,
                    plusTooltip,
                })}
            </Text>
        ),
        [plusTooltip],
    );

    const login = useMemo(
        () => (
            <NoSsr
                fallback={
                    <Link
                        target="_self"
                        rel="noopener noreferrer"
                        url="#"
                        className={cx(deviceMods('link', deviceType))}
                    >
                        {i18nBlock.login()}
                    </Link>
                }
            >
                <Link
                    target="_self"
                    rel="noopener noreferrer"
                    url={getPassportUrl({
                        mode: EPassportMode.AUTH,
                        passportHost: userInfo.passportPath,
                        retpath:
                            typeof window === 'undefined'
                                ? '#'
                                : window.location.href,
                    })}
                    className={cx(deviceMods('link', deviceType))}
                >
                    {i18nBlock.login()}
                </Link>
            </NoSsr>
        ),
        [userInfo, deviceType],
    );

    const plusDescription = useMemo(() => {
        if (isAuthUser(userInfo) && userInfo.plusInfo?.hasPlus)
            return insertJSXIntoKey(
                i18nBlock.plusCashbackAuthorizedHavePlusDescription,
            )({
                plus,
                limitInfo: plusLimitInfo,
            });
        if (isAuthUser(userInfo))
            return insertJSXIntoKey(
                i18nBlock.plusCashbackAuthorizedDescription,
            )({
                plus,
                limitInfo: plusLimitInfo,
            });

        return insertJSXIntoKey(i18nBlock.plusCashbackUnauthorizedDescription)({
            plus,
            login,
            limitInfo: plusLimitInfo,
        });
    }, [login, plus, plusLimitInfo, userInfo]);

    const advantages = useMemo((): TAdvantage[] => {
        const iconWidth = 220;
        const iconHeight = 180;

        return [
            {
                description: plusDescription,
                IconComponent: PlusCashback3DImage,
                iconWidth,
                iconHeight,
            },
            {
                description: i18nRebranding.toursNoExtraChargesDescription(),
                IconComponent: Fair3DImage,
                iconWidth,
                iconHeight,
            },
            {
                description: i18nRebranding.toursManyVariantsDescription(),
                IconComponent: Tours3DImage,
                iconWidth,
                iconHeight,
            },
            {
                description: i18nRebranding.commonSupportDescription(),
                IconComponent: Support3DImage,
                iconWidth,
                iconHeight,
            },
        ];
    }, [plusDescription]);

    const title = useTitle(i18nRebranding.toursTitle);

    return (
        <Advantages
            className={className}
            title={title}
            advantages={advantages}
            {...prepareQaAttributes(props)}
        />
    );
};

export default React.memo(ToursAdvantages);
