import React, {useEffect, useMemo, useRef} from 'react';
import {ParsedQuery} from 'query-string';
import {useDispatch, useSelector} from 'react-redux';

import {EProjectName} from 'constants/common';
import {URLs} from 'constants/urls';

import {ServerDataFetcherBag} from 'server/redux/types';
import {ELoadableReducer} from 'types/common/ELoadableReducer';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';
import {ESubscriptionVerticalName} from 'types/subscription/ESubscriptionVerticalName';
import {
    ESearchFormFieldName,
    TSearchFormErrors,
} from 'components/SearchForm/types';
import {EIndexGoal} from 'utilities/metrika/types/goals';
import {ITrainsSuggest} from 'types/trains/common/ITrainsApiSuggest';
import {ECommonGoal} from 'utilities/metrika/types/goals/common';

import crossLinksReducer from 'reducers/common/crossLinks/reducer';
import {fetchCrossLinks} from 'reducers/common/crossLinks/thunk';
import searchSuggestsReducer from 'reducers/trains/searchSuggests/reducer';
import searchContextReducer from 'reducers/trains/context/reducer';
import previousSearchesReducer from 'reducers/trains/previousSearches/reducer';
import {resetCrossLinks} from 'reducers/common/crossLinks/actions';
import priceCalendarReducer from 'reducers/trains/priceCalendar/reducer';

import crossLinksSelector from 'selectors/common/crossLinksSelector';
import nonceSelector from 'selectors/common/nonceSelector';

import indexTrainsSaga from 'sagas/index/trains/rootSaga';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useServerDataFetcher from 'utilities/hooks/useServerDataFetcher';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {useExperiments} from 'utilities/hooks/useExperiments';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import prepareCrossLinksForGallery from 'utilities/prepareCrossLinksForGallery';

import * as i18nTrainsIndexBlock from 'i18n/trains-indexTexts';
import * as i18nComponents from 'i18n/components';
import * as i18nPopularDirections from 'i18n/trains-popularDirections';

import withSaga from 'containers/withSaga/withSaga';
import withReducers from 'containers/withReducers/withReducers';

import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';
import TrainsAdvantages from 'projects/trains/components/TrainsAdvantages/TrainsAdvantages';
import SearchForm from 'projects/trains/components/SearchForm/SearchForm';
import PreviousSearches from './components/PreviousSearches/PreviousSearches';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';
import {ISuggestValue} from 'components/SearchSuggest/SearchSuggest';
import IndexSubscriptionForm from '../../components/IndexSubscriptionForm/IndexSubscriptionForm';
import CrossLinksGallery from 'projects/index/IndexApp/components/CrossLinksGallery/CrossLinksGallery';
import Text from 'components/Text/Text';
import HowToBuyTrainsTicket from 'projects/index/IndexApp/components/IndexTrains/components/HowToBuyTrainsTicket/HowToBuyTrainsTicket';
import HowToBuyTrainsTicketAtBestPrice from 'projects/index/IndexApp/components/IndexTrains/components/HowToBuyTrainsTicketAtBestPrice/HowToBuyTrainsTicketAtBestPrice';

import trainsServerPreFillSearchForm from 'server/redux/trains/trainsServerPreFillSearchForm';
import prefetchCrossLinks from 'server/redux/common/fetchCrossLinks';

import cx from './IndexTrains.scss';

const ROOT_QA = 'indexTrainsPage';

const unknownFromError =
    i18nComponents.railwaysSearchFormDotValidateDotUnknownFromDirection();
const unknownToError =
    i18nComponents.railwaysSearchFormDotValidateDotUnknownToDirection();

async function prefetchData(
    serverFetcherData: ServerDataFetcherBag,
): Promise<void> {
    await Promise.all([
        trainsServerPreFillSearchForm(serverFetcherData),
        prefetchCrossLinks(serverFetcherData, EProjectName.TRAINS),
    ]);
}

interface IIndexTrainsProps {
    query?: ParsedQuery;
}

const IndexTrains: React.FC<IIndexTrainsProps> = props => {
    const {query = {}} = props;

    const deviceType = useDeviceType();
    const dispatch = useDispatch();
    const crossLinksState = useSelector(crossLinksSelector);
    const {
        project,
        isLoading: isRecipesLoading,
        isFetched: isRecipesFetched,
    } = crossLinksState;
    const galleryItems = useMemo(() => {
        return prepareCrossLinksForGallery(
            crossLinksState,
            EProjectName.TRAINS,
        );
    }, [crossLinksState]);

    const nonce = useSelector(nonceSelector);
    const searchFormFieldsRef = useRef<IFormFieldsRef>();
    const prevFormErrors = useRef<TSearchFormErrors>({});
    const needToRender = useServerDataFetcher([prefetchData]);
    const {rebrandingSubscriptions} = useExperiments();

    const onChangeFromPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<ITrainsSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.TRAINS_MORDA_SEARCH_FORM_CLEAR_FROM);
            }
        },
    );

    const onChangeToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<ITrainsSuggest>) => {
            if (!fieldValue.inputValue) {
                reachGoal(EIndexGoal.TRAINS_MORDA_SEARCH_FORM_CLEAR_TO);
            }
        },
    );

    const onValidationFailed = useImmutableCallback(
        (formErrors: TSearchFormErrors) => {
            if (
                formErrors.from?.includes(unknownFromError) &&
                !prevFormErrors.current.from?.includes(unknownFromError)
            ) {
                reachGoal(EIndexGoal.TRAINS_MORDA_SEARCH_FORM_UNKNOWN_FROM);
            }

            if (
                formErrors.to?.includes(unknownToError) &&
                !prevFormErrors.current.to?.includes(unknownToError)
            ) {
                reachGoal(EIndexGoal.TRAINS_MORDA_SEARCH_FORM_UNKNOWN_TO);
            }

            prevFormErrors.current = formErrors;
        },
    );

    const onResetCalendar = useReachGoal(
        EIndexGoal.TRAINS_MORDA_SEARCH_FORM_RESET_DATE,
    );

    useEffect(() => {
        return (): void => {
            dispatch(resetCrossLinks());
        };
    }, [dispatch]);

    useEffect(() => {
        if (project !== EProjectName.TRAINS || !isRecipesFetched) {
            dispatch(fetchCrossLinks(EProjectName.TRAINS));
        }
    }, [dispatch, isRecipesFetched, project]);

    // Параметры m__outdate_search_request и m__outdate_order_request появляются в query при редиректе
    // cо страниц поиска и заказа с датами в прошлом.
    // query.m__outdate_index_request - при редиректе с главной страницы, если в query есть дата в прошлом.
    const isOutDateRequest = Boolean(
        query.m__outdate_search_request ||
            query.m__outdate_order_request ||
            query.m__outdate_index_request,
    );

    const crossLinkGalleryTitle = useMemo(() => {
        return insertJSXIntoKey(i18nTrainsIndexBlock.crossLinkGalleryTitle)({
            start: (
                <Text weight="fat" size="inherit">
                    {i18nTrainsIndexBlock.crossLinkGalleryTitleStart()}
                </Text>
            ),
            end: (
                <Text size="inherit" whiteSpace="nowrap">
                    {i18nTrainsIndexBlock.crossLinkGalleryTitleEnd()}
                </Text>
            ),
        });
    }, []);

    if (!needToRender) {
        return null;
    }

    // По мотивам https://st.yandex-team.ru/TRAVELFRONT-7169
    const renderCrossaleBlock = project === EProjectName.TRAINS;

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), {
                root_rebrandingSubscriptions: rebrandingSubscriptions,
            })}
            {...prepareQaAttributes(ROOT_QA)}
        >
            <SearchForm
                className={cx('searchForm')}
                needToSetFromSuggestByGeoPosition
                query={query}
                initiallyCalendarIsOpen={
                    isOutDateRequest && !deviceType.isMobile
                }
                autoFocusFieldName={
                    query?.focus as ESearchFormFieldName | undefined
                }
                fieldsRef={searchFormFieldsRef}
                onChangeFromPoint={onChangeFromPoint}
                onChangeToPoint={onChangeToPoint}
                onValidationFailed={onValidationFailed}
                onResetCalendar={onResetCalendar}
            />

            {deviceType.isDesktop && (
                <PreviousSearches
                    className={cx('previousSearches')}
                    searchFormFieldsRef={searchFormFieldsRef}
                />
            )}

            <AdFoxBanner
                className={cx('banner')}
                fixed
                type={EAdFoxBannerType.Inline}
                position={EAdFoxBannerPosition.Top}
            />

            <TrainsAdvantages className={cx('advantages')} />

            <IndexSubscriptionForm
                className={cx('subscribe')}
                vertical={ESubscriptionVerticalName.Trains}
            />

            {renderCrossaleBlock && (
                <CrossLinksGallery
                    className={cx('recipes')}
                    title={crossLinkGalleryTitle}
                    items={galleryItems}
                    loading={isRecipesLoading}
                    moreUrl={URLs.trainsPopularRoutes}
                    nonce={nonce}
                    markupText={i18nPopularDirections.markupDashText()}
                    itemClickMetrikaGoal={ECommonGoal.RECIPE_LINK}
                    {...prepareQaAttributes({
                        parent: ROOT_QA,
                        current: 'crossLinksGallery',
                    })}
                />
            )}

            <HowToBuyTrainsTicket className={cx('howToBuyATicket')} />

            <HowToBuyTrainsTicketAtBestPrice className={cx('texts')} />
        </div>
    );
};

export default withReducers([
    [ELoadableReducer.TRAINS_SEARCH_SUGGESTS, searchSuggestsReducer],
    [ELoadableReducer.TRAINS_CONTEXT, searchContextReducer],
    [ELoadableReducer.TRAINS_PREVIOUS_SEARCHES, previousSearchesReducer],
    [ELoadableReducer.COMMON_CROSSLINKS, crossLinksReducer],
    [ELoadableReducer.TRAINS_PRICE_CALENDAR, priceCalendarReducer],
])(withSaga([indexTrainsSaga])(IndexTrains));
